#include <solomon/libs/cpp/ts_codec/double_ts_codec.h>
#include <solomon/libs/cpp/ts_codec/ut/helpers.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace NSolomon::NTs;

class TDoubleTsCodecTest: public ::testing::Test {
public:
    void SetUp() override {
        Buf_ = {};
    }

    void Encode(std::vector<TDoublePoint> points, TColumnSet columns = TDoublePoint::AggrColumns) {
        Points_ = std::move(points);

        TBitWriter w{&Buf_};
        auto encoder = TDoubleTsEncoder{columns, &w};
        for (const auto& p: Points_) {
            encoder.EncodePoint(p);
        }

        encoder.Flush();
    }

    void DecodeAndCheck(TColumnSet columns = TDoublePoint::AggrColumns, TDoublePoint point = {}) {
        auto decoder = TDoubleTsDecoder{columns, Buf_};
        ASSERT_TRUE(decoder.HasNext());

        for (size_t i = 0; i < Points_.size(); ++i) {
            const auto& expected = Points_[i];

            bool hasNext;
            ASSERT_NO_THROW({ hasNext = decoder.NextPoint(&point); }) << "point #" << i;
            ASSERT_TRUE(hasNext) << "point #" << i;

            EXPECT_EQ(expected.Time, point.Time) << "point #" << i;
            EXPECT_EQ(expected.Step, point.Step) << "point #" << i;
            EXPECT_EQ(expected.Num, point.Num) << "point #" << i;
            EXPECT_EQ(expected.Denom, point.Denom) << "point #" << i;

            EXPECT_EQ(expected.Merge, point.Merge) << "point #" << i;
            EXPECT_EQ(expected.Count, point.Count) << "point #" << i;
        }

        ASSERT_FALSE(decoder.HasNext());
    }

    static TDoublePoint MakePoint(TStringBuf time, double value, ui64 stepMillis = 0, ui64 count = 0) {
        TDoublePoint point;
        point.Time = TInstant::ParseIso8601(time);
        point.Num = value;

        if (stepMillis) {
            point.Step = TDuration::MilliSeconds(stepMillis);
        }

        if (count) {
            point.Merge = true;
            point.Count = count;
        }
        return point;
    }

private:
    std::vector<TDoublePoint> Points_;
    TBitBuffer Buf_;
};

TEST_F(TDoubleTsCodecTest, SequentialGrowAsInteger) {
    Encode({
            MakePoint("2020-05-30T11:12:00Z", 1),
            MakePoint("2020-05-30T11:12:15Z", 2),
            MakePoint("2020-05-30T11:12:30Z", 3),
            MakePoint("2020-05-30T11:12:45Z", 4),
            MakePoint("2020-05-30T11:13:00Z", 5),
    });
    DecodeAndCheck();
}

TEST_F(TDoubleTsCodecTest, SequentialGrowAsDouble) {
    Encode({
            MakePoint("2020-05-30T11:12:00Z", 1.5),
            MakePoint("2020-05-30T11:12:15Z", 2.0),
            MakePoint("2020-05-30T11:12:30Z", 2.5),
            MakePoint("2020-05-30T11:12:45Z", 3.0),
            MakePoint("2020-05-30T11:13:00Z", 3.5),
            MakePoint("2020-05-30T11:13:15Z", 4.0),
    });
    DecodeAndCheck();
}

TEST_F(TDoubleTsCodecTest, SequentialButInRandomOrder) {
    Encode({
            MakePoint("2020-05-30T11:12:00Z", 2),
            MakePoint("2020-05-30T11:12:15Z", 5),
            MakePoint("2020-05-30T11:12:30Z", 3),
            MakePoint("2020-05-30T11:12:45Z", 1),
            MakePoint("2020-05-30T11:13:00Z", 4),
    });
    DecodeAndCheck();
}

TEST_F(TDoubleTsCodecTest, SameValue) {
    Encode({
            MakePoint("2020-05-30T11:12:00Z", 3.14),
            MakePoint("2020-05-30T11:12:15Z", 3.14),
            MakePoint("2020-05-30T11:12:30Z", 3.14),
            MakePoint("2020-05-30T11:12:45Z", 3.14),
            MakePoint("2020-05-30T11:13:00Z", 3.14),
    });
    DecodeAndCheck();
}

TEST_F(TDoubleTsCodecTest, Spikes) {
    Encode({
            MakePoint("2020-05-30T11:12:00Z", 1),
            MakePoint("2020-05-30T11:12:15Z", 2),
            MakePoint("2020-05-30T11:12:30Z", 1'123'123.0),
            MakePoint("2020-05-30T11:12:45Z", 500),
            MakePoint("2020-05-30T11:13:00Z", 3),
    });
    DecodeAndCheck();
}

TEST_F(TDoubleTsCodecTest, PositiveAndNegative) {
    Encode({
            MakePoint("2020-05-30T11:12:00Z", 511),
            MakePoint("2020-05-30T11:12:15Z", -123),
            MakePoint("2020-05-30T11:12:30Z", 0),
            MakePoint("2020-05-30T11:12:45Z", 9123.5),
            MakePoint("2020-05-30T11:13:00Z", -333),
    });
    DecodeAndCheck();
}

TEST_F(TDoubleTsCodecTest, NotStableInterval) {
    Encode({
            MakePoint("2020-05-30T11:12:01Z", 1.5),
            MakePoint("2020-05-30T11:12:13Z", 2.0),
            MakePoint("2020-05-30T11:12:21Z", 2.5),
            MakePoint("2020-05-30T11:12:35Z", 3.0),
            MakePoint("2020-05-30T11:13:00Z", 3.5),
    });
    DecodeAndCheck();
}

TEST_F(TDoubleTsCodecTest, SameTime) {
    Encode({
            MakePoint("2020-05-30T11:12:00Z", 1.5),
            MakePoint("2020-05-30T11:12:00Z", 2.0),
            MakePoint("2020-05-30T11:12:00Z", 2.5),
            MakePoint("2020-05-30T11:12:00Z", 3.0),
            MakePoint("2020-05-30T11:12:00Z", 3.5),
    });
    DecodeAndCheck();
}

TEST_F(TDoubleTsCodecTest, ConstantStep) {
    Encode({
            MakePoint("2020-05-30T11:12:00Z", 1.5, 15'000),
            MakePoint("2020-05-30T11:12:15Z", 2.0, 15'000),
            MakePoint("2020-05-30T11:12:30Z", 2.5, 15'000),
            MakePoint("2020-05-30T11:12:45Z", 3.0, 15'000),
            MakePoint("2020-05-30T11:13:00Z", 3.5, 15'000),
            MakePoint("2020-05-30T11:13:15Z", 4.0, 15'000),
    });
    DecodeAndCheck();
}

TEST_F(TDoubleTsCodecTest, ChangingStep) {
    Encode({
            MakePoint("2020-05-30T11:12:00Z", 1.5, 15'000),
            MakePoint("2020-05-30T11:12:15Z", 2.0, 30'000),
            MakePoint("2020-05-30T11:12:30Z", 2.5, 50'000),
            MakePoint("2020-05-30T11:12:45Z", 3.0, 15'000),
            MakePoint("2020-05-30T11:13:00Z", 3.5, 15'000),
            MakePoint("2020-05-30T11:13:15Z", 4.0, 40'000),
    });
    DecodeAndCheck();
}

TEST_F(TDoubleTsCodecTest, ConstantCount) {
    Encode({
            MakePoint("2020-05-30T11:12:00Z", 1.5, 0, 10),
            MakePoint("2020-05-30T11:12:15Z", 2.0, 0, 10),
            MakePoint("2020-05-30T11:12:30Z", 2.5, 0, 10),
            MakePoint("2020-05-30T11:12:45Z", 3.0, 0, 10),
            MakePoint("2020-05-30T11:13:00Z", 3.5, 0, 10),
            MakePoint("2020-05-30T11:13:15Z", 4.0, 0, 10),
    });
    DecodeAndCheck();
}

TEST_F(TDoubleTsCodecTest, ChanginCount) {
    Encode({
            MakePoint("2020-05-30T11:12:00Z", 1.5, 0, 5),
            MakePoint("2020-05-30T11:12:15Z", 2.0, 0, 3),
            MakePoint("2020-05-30T11:12:30Z", 2.5, 0, 0),
            MakePoint("2020-05-30T11:12:45Z", 3.0, 0, 1),
            MakePoint("2020-05-30T11:13:00Z", 3.5, 0, 2),
            MakePoint("2020-05-30T11:13:15Z", 4.0, 0, 3),
    });
    DecodeAndCheck();
}

TEST_F(TDoubleTsCodecTest, DirtyPoint) {
    Encode({
            MakePoint("2020-05-30T11:12:00Z", 42),
            MakePoint("2020-05-30T11:12:15Z", 42),
            MakePoint("2020-05-30T11:12:30Z", 42),
            MakePoint("2020-05-30T11:12:45Z", 42),
            MakePoint("2020-05-30T11:13:00Z", 42),
    }, TDoublePoint::SimpleColumns);

    TDoublePoint point;

    // All of these should be overwritten when decoding
    point.Time = TInstant::Minutes(10);
    point.Step = TDuration::Minutes(30);
    point.Count = 10;
    point.Merge = true;

    DecodeAndCheck(TDoublePoint::SimpleColumns, point);
}
