#include <solomon/libs/cpp/ts_codec/frame_codec.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace NSolomon::NTs;

TEST(TFrameCodecTest, InitState) {
    TFrameEncoder encoder;
    EXPECT_EQ(encoder.FrameIdx(), 0u);
}

TEST(TFrameCodecTest, InitFrame) {
    TBitBuffer buf;
    TBitWriter w{&buf};

    TFrameEncoder encoder;
    encoder.InitFrame(&w);

    EXPECT_EQ(encoder.FrameIdx(), 0u);
    EXPECT_EQ(w.Pos(), 40u);
}

TEST(TFrameCodecTest, DecodeEmptyFrame) {
    TBitBuffer buf;
    TBitWriter w{&buf};

    {
        TFrameEncoder encoder;
        encoder.InitFrame(&w);
    }

    TBitReader r{buf};
    TFrameDecoder decoder;

    ASSERT_TRUE(decoder.Next(&r));
    EXPECT_EQ(decoder.PayloadIndex(), FRAME_HEADER_SIZE);
    EXPECT_EQ(decoder.PayloadBits(), 0u);

    EXPECT_FALSE(decoder.HasFooter());
    ASSERT_FALSE(decoder.Next(&r));
}

TEST(TFrameCodecTest, DecodeFrame) {
    TBitBuffer buf;
    TBitWriter w{&buf};

    {
        TFrameEncoder encoder;
        encoder.InitFrame(&w);

        // frame data
        w.WriteBit(true);
        w.WriteInt32(42);
        w.WriteBit(false);
    }

    TBitReader r{buf};
    TFrameDecoder decoder;

    ASSERT_TRUE(decoder.Next(&r));

    EXPECT_EQ(decoder.PayloadIndex(), FRAME_HEADER_SIZE);
    EXPECT_EQ(decoder.PayloadBits(), 34u);

    r.SetPos(decoder.PayloadIndex());
    ASSERT_TRUE(r.ReadBit());
    ASSERT_EQ(r.ReadInt32(), 42u);
    ASSERT_FALSE(r.ReadBit());

    EXPECT_FALSE(decoder.HasFooter());
    ASSERT_EQ(r.Left(), 0u);

    // no more frames
    ASSERT_FALSE(decoder.Next(&r));
}

TEST(TFrameCodecTest, DecodeFrameWithFooter) {
    TBitBuffer buf;
    TBitWriter w{&buf};

    {
        TFrameEncoder encoder;
        encoder.InitFrame(&w);

        // frame payload
        w.WriteBit(true);
        w.WriteInt32(42);
        w.WriteBit(false);

        encoder.FinishFrame(&w, []() {
            // empty footer payload
        });

        w.Flush();
    }


    TBitReader r{buf};
    TFrameDecoder decoder;

    ASSERT_TRUE(decoder.Next(&r));

    EXPECT_EQ(decoder.PayloadIndex(), FRAME_HEADER_SIZE);
    EXPECT_EQ(decoder.PayloadBits(), 34u);

    r.SetPos(decoder.PayloadIndex());
    ASSERT_TRUE(r.ReadBit());
    ASSERT_EQ(r.ReadInt32(), 42u);
    ASSERT_FALSE(r.ReadBit());

    EXPECT_TRUE(decoder.HasFooter());
    EXPECT_EQ(decoder.FooterIndex(), FRAME_HEADER_SIZE + AlignToByte(33u) + FRAME_FOOTER_SIZE);
    EXPECT_EQ(decoder.FooterBits(), 0u);

    r.SetPos(decoder.FooterIndex());
    ASSERT_EQ(r.Left(), 0u);

    // no more frames
    ASSERT_FALSE(decoder.Next(&r));
}

TEST(TFrameCodecTest, DecodeManyFrames) {
    TBitBuffer buf;
    TBitWriter w{&buf};
    TFrameEncoder encoder;

    // frame #1
    {
        encoder.InitFrame(&w);
        // frame payload
        w.WriteBit(true);
        w.WriteInt32(42);

        encoder.FinishFrame(&w, []() {
            // empty footer payload
        });
    }

    // frame #2
    {
        encoder.InitFrame(&w);
        // frame payload
        w.WriteBit(false);
        w.WriteInt32(43);
        w.WriteBit(false);
        w.WriteInt32(44);

        encoder.FinishFrame(&w, [&w]() {
            // footer payload
            w.WriteInt64(45);
            w.WriteBit(true);
        });
    }

    // frame #3
    {
        encoder.InitFrame(&w);
        // frame payload
        w.WriteBit(true);
    }

    w.Flush();

    TBitReader r{buf};
    TFrameDecoder decoder;

    // frame #1
    {
        size_t startPos = r.Pos();
        ASSERT_TRUE(decoder.Next(&r));

        EXPECT_EQ(decoder.PayloadIndex(), startPos + FRAME_HEADER_SIZE);
        EXPECT_EQ(decoder.PayloadBits(), 33u);

        r.SetPos(decoder.PayloadIndex());
        ASSERT_TRUE(r.ReadBit());
        ASSERT_EQ(r.ReadInt32(), 42u);

        EXPECT_TRUE(decoder.HasFooter());
        EXPECT_EQ(decoder.FooterIndex(), startPos + FRAME_HEADER_SIZE + AlignToByte(33u) + FRAME_FOOTER_SIZE);
        EXPECT_EQ(decoder.FooterBits(), 0u);

        r.SetPos(decoder.FooterIndex());
        // nothing to read
    }

    // frame #2
    {
        size_t startPos = r.Pos();
        ASSERT_TRUE(decoder.Next(&r));

        EXPECT_EQ(decoder.PayloadIndex(), startPos + FRAME_HEADER_SIZE);
        EXPECT_EQ(decoder.PayloadBits(), 66u);

        r.SetPos(decoder.PayloadIndex());
        ASSERT_FALSE(r.ReadBit());
        ASSERT_EQ(r.ReadInt32(), 43u);
        ASSERT_FALSE(r.ReadBit());
        ASSERT_EQ(r.ReadInt32(), 44u);

        EXPECT_TRUE(decoder.HasFooter());
        EXPECT_EQ(decoder.FooterIndex(), startPos + FRAME_HEADER_SIZE + AlignToByte(66u) + FRAME_FOOTER_SIZE);
        EXPECT_EQ(decoder.FooterBits(), AlignToByte(65u));

        r.SetPos(decoder.FooterIndex());
        ASSERT_EQ(r.ReadInt64(), 45u);
        ASSERT_TRUE(r.ReadBit());

        // skip alignment padding
        r.SetPos(decoder.FooterIndex() + AlignToByte(65u));
    }

    // frame #3
    {
        size_t startPos = r.Pos();
        ASSERT_TRUE(decoder.Next(&r));

        EXPECT_EQ(decoder.PayloadIndex(), startPos + FRAME_HEADER_SIZE);
        EXPECT_EQ(decoder.PayloadBits(), 1u);

        r.SetPos(decoder.PayloadIndex());
        ASSERT_TRUE(r.ReadBit());
        EXPECT_FALSE(decoder.HasFooter());
    }

    ASSERT_FALSE(decoder.Next(&r));
}
