#include "helpers.h"

#include <solomon/libs/cpp/ts_codec/bit_reader.h>
#include <solomon/libs/cpp/ts_codec/bit_writer.h>

#include <util/random/random.h>
#include <util/string/builder.h>
#include <util/string/hex.h>

namespace NSolomon::NTs {

TBuffer ParseBin(TStringBuf str) {
    return ParseBin2(str).first;
}

std::pair<TBuffer, size_t> ParseBin2(TStringBuf str) {
    TBitBuffer buf;
    TBitWriter w{&buf};
    for (char ch: str) {
        if (ch == ' ') {
            continue;
        } else if (ch == '0') {
            w.WriteBit(false);
        } else if (ch == '1') {
            w.WriteBit(true);
        } else {
            ythrow yexception() << "unexpected char: " << ch;
        }
    }
    w.Flush();
    return {std::move(buf).Container(), w.Pos()};
}

TString ToBin(const TBitBuffer& buf) {
    TStringBuilder sb;
    TBitReader r{buf};

    for (int i = 1; r.Left() != 0; ++i) {
        sb.append(r.ReadBit() ? '1' : '0');
        if (i % 8 == 0) {
            sb.append(' ');
        }
    }
    return sb;
}

TString ToBin(const TBuffer& buf) {
    TStringBuilder sb;
    TBitReader r{buf.data(), buf.size() * 8};

    for (int i = 1; r.Left() != 0; ++i) {
        sb.append(r.ReadBit() ? '1' : '0');
        if (i % 8 == 0) {
            sb.append(' ');
        }
    }
    return sb;
}

TString ToHex(const TBitBuffer& buf) {
    return HexEncode(buf.Data(), ByteCount(buf.Size()));
}

TString ToHex(const TBuffer& buf) {
    return HexEncode(buf.data(), buf.size());
}

void FillRandomly(size_t max, TBitBuffer* buf) {
    for (size_t i = 1 + RandomNumber<size_t>(max); i != 0; i--) {
        buf->PushBack(RandomNumber<bool>());
    }
}

::testing::AssertionResult IsSameBuf(const TBitBuffer& actualBuf, TStringBuf expectedBits) {
    TBitReader r{actualBuf};

    size_t pos = 0;
    for (size_t i = 0; i < expectedBits.size(); ++i) {
        char expectedCh = expectedBits[i];
        if (expectedCh == ' ') continue;

        if (r.Left() == 0) {
            return ::testing::AssertionFailure()
                << "buffer is shorter than expected (buf: " << ToBin(actualBuf)
                << ", expected buf: " << expectedBits << ')';
        }

        ++pos;

        char ch = r.ReadBit() ? '1' : '0';
        if (ch != expectedCh) {
            return ::testing::AssertionFailure()
                << "unexpected bit at " << pos << ", expected: " << expectedCh << ", but got " << ch
                << " (buf: " << ToBin(actualBuf) << ", expected buf: " << expectedBits << ')';
        }
    }

    if (r.Left() < 8) {
        while (r.Left() != 0 && !r.ReadBit()) /* skip trailing zeros in the last byte */ ;

        if (r.Left() == 0) {
            return ::testing::AssertionSuccess();
        }
    }

    return ::testing::AssertionFailure()
        << "buffer is longer than expected (buf: " << ToBin(actualBuf)
        << ", expected buf: " << expectedBits << ')';
}

} // namespace NSolomon::NTs
