#include <solomon/libs/cpp/ts_codec/hist_log_ts_codec.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace NSolomon::NTs;

class TLogHistogramTsCodecTest: public ::testing::Test {
public:
    void SetUp() override {
        Points_.clear();
        Buf_ = {};
        Writer_ = std::make_unique<TBitWriter>(&Buf_);
        Encoder_ = std::make_unique<TLogHistogramTsEncoder>(TLogHistogramPoint::AggrColumns, Writer_.get());
    }

    void Encode(TStringBuf time, NValue::TLogHistogram hist) {
        Points_.emplace_back();
        auto& point = Points_.back();
        point.Time = TInstant::ParseIso8601(time);
        point.ZeroCount = hist.ZeroCount;
        point.StartPower = hist.StartPower;
        point.MaxBucketCount = hist.MaxBucketCount;
        point.Base = hist.Base;
        point.Values = std::move(hist).Values;

        Encoder_->EncodePoint(point);
    }

    void DecodeAndCheck() {
        Encoder_->Flush();
        auto decoder = TLogHistogramTsDecoder::Aggr(Buf_);
        ASSERT_TRUE(decoder.HasNext());

        TLogHistogramPoint point;
        for (size_t i = 0; i < Points_.size(); ++i) {
            const auto& expected = Points_[i];

            bool hasNext;
            ASSERT_NO_THROW({ hasNext = decoder.NextPoint(&point); }) << "point #" << i;
            ASSERT_TRUE(hasNext) << "point #" << i;

            // common fields
            EXPECT_EQ(expected.Time, point.Time) << "point #" << i;
            EXPECT_EQ(expected.Step, point.Step) << "point #" << i;

            // log histogram fields
            EXPECT_EQ(expected.Values, point.Values) << "point #" << i;
            EXPECT_EQ(expected.ZeroCount, point.ZeroCount) << "point #" << i;
            EXPECT_EQ(expected.StartPower, point.StartPower) << "point #" << i;
            EXPECT_EQ(expected.MaxBucketCount, point.MaxBucketCount) << "point #" << i;
            EXPECT_EQ(expected.Base, point.Base) << "point #" << i;

            // aggregate fields
            EXPECT_EQ(expected.Merge, point.Merge) << "point #" << i;
            EXPECT_EQ(expected.Count, point.Count) << "point #" << i;
        }

        ASSERT_FALSE(decoder.HasNext());
    }

private:
    std::unique_ptr<TBitWriter> Writer_;
    std::unique_ptr<TLogHistogramTsEncoder> Encoder_;
    std::vector<TLogHistogramPoint> Points_;
    TBitBuffer Buf_;
};

TEST_F(TLogHistogramTsCodecTest, One) {
    Encode("2020-05-30T11:12:00Z", NValue::TLogHistogram{
            .Values={1.1, 2.2, 3.3, 4.4},
            .ZeroCount=2,
            .StartPower=-1,
    });
    DecodeAndCheck();
}

TEST_F(TLogHistogramTsCodecTest, SameStartPower) {
    Encode("2020-05-30T11:12:00Z", NValue::TLogHistogram{
            .Values={1.1, 2.2, 3.3, 4.4},
            .ZeroCount=2,
            .StartPower=3,
    });
    Encode("2020-05-30T11:12:15Z", NValue::TLogHistogram{
            .Values={3.3, 4.4, 5.5},
            .ZeroCount=10,
            .StartPower=3,
    });
    DecodeAndCheck();
}

TEST_F(TLogHistogramTsCodecTest, SameZeroCount) {
    Encode("2020-05-30T11:12:00Z", NValue::TLogHistogram{
            .Values={1.1, 2.2, 3.3, 4.4},
            .ZeroCount=10,
            .StartPower=1,
    });
    Encode("2020-05-30T11:12:15Z", NValue::TLogHistogram{
            .Values={3.3, 4.4, 5.5},
            .ZeroCount=0,
            .StartPower=3,
    });
    DecodeAndCheck();
}

TEST_F(TLogHistogramTsCodecTest, DifferentMaxBuckets) {
    Encode("2020-05-30T11:12:00Z", NValue::TLogHistogram{
            .Values={1.1, 2.2, 3.3, 4.4},
            .ZeroCount=10,
            .StartPower=1,
            .MaxBucketCount=10
    });
    Encode("2020-05-30T11:12:15Z", NValue::TLogHistogram{
            .Values={1.1, 2.2, 3.3, 4.4},
            .ZeroCount=10,
            .StartPower=1,
            .MaxBucketCount=4
    });
    DecodeAndCheck();
}


