#include <solomon/libs/cpp/ts_codec/hist_ts_codec.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace NSolomon::NTs;

class THistogramTsCodecTest: public ::testing::Test {
public:
    void SetUp() override {
        Buf_ = {};
    }

    void Encode(std::vector<THistogramPoint> points) {
        Points_ = std::move(points);

        TBitWriter w{&Buf_};
        auto encoder = THistogramTsEncoder::Aggr(&w);
        for (const auto& p: Points_) {
            encoder.EncodePoint(p);
        }

        encoder.Flush();
    }

    void DecodeAndCheck() {
        auto decoder = THistogramTsDecoder::Aggr(Buf_);
        ASSERT_TRUE(decoder.HasNext());

        THistogramPoint point;
        for (size_t i = 0; i < Points_.size(); ++i) {
            const auto& expected = Points_[i];

            bool hasNext;
            ASSERT_NO_THROW({ hasNext = decoder.NextPoint(&point); }) << "point #" << i;
            ASSERT_TRUE(hasNext) << "point #" << i;

            // common fields
            EXPECT_EQ(expected.Time, point.Time) << "point #" << i;
            EXPECT_EQ(expected.Step, point.Step) << "point #" << i;

            // histogram fields
            EXPECT_EQ(expected.Buckets.size(), point.Buckets.size()) << "point #" << i;
            for (size_t j = 0; j < expected.Buckets.size(); ++j) {
                EXPECT_EQ(expected.Buckets[j].UpperBound, point.Buckets[j].UpperBound)
                        << "point #" << i << ", bucket #" << j;
                EXPECT_EQ(expected.Buckets[j].Value, point.Buckets[j].Value)
                        << "point #" << i << ", bucket #" << j;
            }
            EXPECT_EQ(expected.Denom, point.Denom) << "point #" << i;

            // aggregate fields
            EXPECT_EQ(expected.Merge, point.Merge) << "point #" << i;
            EXPECT_EQ(expected.Count, point.Count) << "point #" << i;
        }

        ASSERT_FALSE(decoder.HasNext());
    }

    static THistogramPoint MakePoint(TStringBuf time, std::vector<NValue::THistogram::TBucket> buckets, ui64 denom = 0) {
        THistogramPoint point;
        point.Time = TInstant::ParseIso8601(time);
        point.Buckets = std::move(buckets);
        point.Denom = denom;
        return point;
    }

private:
    std::vector<THistogramPoint> Points_;
    TBitBuffer Buf_;
};

TEST_F(THistogramTsCodecTest, One) {
    Encode({
            MakePoint("2020-05-30T11:12:00Z", {{10, 0}, {20, 3}, {30, 2}})
    });
    DecodeAndCheck();
}

TEST_F(THistogramTsCodecTest, SameBounds) {
    Encode({
            MakePoint("2020-05-30T11:12:00Z", {{10, 0}, {20, 3}, {30, 2}}),
            MakePoint("2020-05-30T11:12:15Z", {{10, 0}, {20, 3}, {30, 2}}),
            MakePoint("2020-05-30T11:12:30Z", {{10, 4}, {20, 3}, {30, 8}}),
            MakePoint("2020-05-30T11:12:45Z", {{10, 5}, {20, 9}, {30, 8}}),
            MakePoint("2020-05-30T11:13:00Z", {{10, 9}, {20, 9}, {30, 9}}),
    });
    DecodeAndCheck();
}

TEST_F(THistogramTsCodecTest, ExtendBounds) {
    Encode({
            MakePoint("2020-05-30T11:12:00Z", {{10, 1}, {20, 2}, {30, 3}}),
            MakePoint("2020-05-30T11:12:15Z", {{10, 3}, {20, 4}, {30, 5}}),
            MakePoint("2020-05-30T11:12:30Z", {{10, 6}, {20, 7}, {30, 8}}),
            MakePoint("2020-05-30T11:12:45Z", {{10, 5}, {15, 4}, {20, 3}, {25, 2}, {30, 1}}),
            MakePoint("2020-05-30T11:13:00Z", {{10, 1}, {15, 2}, {20, 3}, {25, 4}, {30, 5}}),
            MakePoint("2020-05-30T11:13:15Z", {{10, 3}, {15, 3}, {20, 3}, {25, 3}, {30, 3}}),
    });
    DecodeAndCheck();
}

TEST_F(THistogramTsCodecTest, ShrinkBounds) {
    Encode({
            MakePoint("2020-05-30T11:12:00Z", {{10, 5}, {15, 4}, {20, 3}, {25, 2}, {30, 1}}),
            MakePoint("2020-05-30T11:12:15Z", {{10, 1}, {15, 2}, {20, 3}, {25, 4}, {30, 5}}),
            MakePoint("2020-05-30T11:12:30Z", {{10, 3}, {15, 3}, {20, 3}, {25, 3}, {30, 3}}),
            MakePoint("2020-05-30T11:12:45Z", {{10, 1}, {20, 2}, {30, 3}}),
            MakePoint("2020-05-30T11:13:00Z", {{10, 3}, {20, 4}, {30, 5}}),
            MakePoint("2020-05-30T11:13:15Z", {{10, 6}, {20, 7}, {30, 8}}),
    });
    DecodeAndCheck();
}

TEST_F(THistogramTsCodecTest, DifferentBounds) {
    Encode({
            MakePoint("2020-05-30T11:12:00Z", {{10, 1}, {20, 2}}),
            MakePoint("2020-05-30T11:12:15Z", {{10, 3}, {20, 4}, {30, 5}}),
            MakePoint("2020-05-30T11:12:30Z", {{10, 5}, {20, 6}}),
            MakePoint("2020-05-30T11:12:45Z", {{10, 5}, {15, 4}, {20, 3}, {25, 2}, {30, 1}}),
            MakePoint("2020-05-30T11:13:00Z", {{20, 3}}),
            MakePoint("2020-05-30T11:13:15Z", {{15, 6}, {30, 7}}),
    });
    DecodeAndCheck();
}

TEST_F(THistogramTsCodecTest, DoubleBounds) {
    Encode({
            MakePoint("2020-05-30T11:12:00Z", {{1.41, 0}, {2.71, 3}, {3.14, 2}}),
            MakePoint("2020-05-30T11:12:15Z", {{1.41, 0}, {2.71, 3}, {3.14, 2}}),
            MakePoint("2020-05-30T11:12:30Z", {{1.41, 4}, {2.71, 3}, {3.14, 8}}),
            MakePoint("2020-05-30T11:12:45Z", {{1.41, 5}, {2.71, 9}, {3.14, 8}}),
            MakePoint("2020-05-30T11:13:00Z", {{1.41, 9}, {2.71, 9}, {3.14, 9}}),
    });
    DecodeAndCheck();
}

TEST_F(THistogramTsCodecTest, Denominator) {
    Encode({
            MakePoint("2020-05-30T11:12:00Z", {{1.41, 0}, {2.71, 3}, {3.14, 2}}),
            MakePoint("2020-05-30T11:12:15Z", {{1.41, 0}, {2.71, 3}, {3.14, 2}}, 15'000),
            MakePoint("2020-05-30T11:12:30Z", {{1.41, 4}, {2.71, 3}, {3.14, 8}}, 30'000),
            MakePoint("2020-05-30T11:12:45Z", {{1.41, 5}, {2.71, 9}, {3.14, 8}}, 15'000),
            MakePoint("2020-05-30T11:13:00Z", {{1.41, 9}, {2.71, 9}, {3.14, 9}}),
    });
    DecodeAndCheck();
}

TEST_F(THistogramTsCodecTest, TrailingEmpty) {
    Encode({
            MakePoint("2020-05-30T11:12:00Z", {{10, 0}, {20, 3}, {30, 2}}),
            MakePoint("2020-05-30T11:12:15Z", {}),
    });
    DecodeAndCheck();
}
