#include <solomon/libs/cpp/ts_codec/summary_double_ts_codec.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace NSolomon::NTs;

class TSummaryDoubleTsCodecTest: public ::testing::Test {
public:
    void SetUp() override {
        Buf_ = {};
    }

    void Encode(std::vector<TSummaryDoublePoint> points) {
        Points_ = std::move(points);

        TBitWriter w{&Buf_};
        auto encoder = TSummaryDoubleTsEncoder::Aggr(&w);
        for (const auto& p: Points_) {
            encoder.EncodePoint(p);
        }

        encoder.Flush();
    }

    void DecodeAndCheck() {
        auto decoder = TSummaryDoubleTsDecoder::Aggr(Buf_);
        ASSERT_TRUE(decoder.HasNext());

        TSummaryDoublePoint point;
        for (size_t i = 0; i < Points_.size(); ++i) {
            const auto& expected = Points_[i];

            bool hasNext;
            ASSERT_NO_THROW({ hasNext = decoder.NextPoint(&point); }) << "point #" << i;
            ASSERT_TRUE(hasNext) << "point #" << i;

            // common fields
            EXPECT_EQ(expected.Time, point.Time) << "point #" << i;
            EXPECT_EQ(expected.Step, point.Step) << "point #" << i;

            // summary fields
            EXPECT_EQ(expected.CountValue, point.CountValue) << "point #" << i;
            EXPECT_EQ(expected.Sum, point.Sum) << "point #" << i;
            EXPECT_EQ(expected.Min, point.Min) << "point #" << i;
            EXPECT_EQ(expected.Max, point.Max) << "point #" << i;
            EXPECT_EQ(expected.Last, point.Last) << "point #" << i;

            // aggregate fields
            EXPECT_EQ(expected.Merge, point.Merge) << "point #" << i;
            EXPECT_EQ(expected.Count, point.Count) << "point #" << i;
        }

        ASSERT_FALSE(decoder.HasNext());
    }

    static TSummaryDoublePoint MakePoint(TStringBuf time, i64 count, double sum, double min, double max, double last) {
        TSummaryDoublePoint point;
        point.Time = TInstant::ParseIso8601(time);
        point.CountValue = count;
        point.Sum = sum;
        point.Min = min;
        point.Max = max;
        point.Last = last;
        return point;
    }

private:
    std::vector<TSummaryDoublePoint> Points_;
    TBitBuffer Buf_;
};

TEST_F(TSummaryDoubleTsCodecTest, One) {
    Encode({ MakePoint("2020-05-30T11:12:00Z", 10, 123.21, 1.3, 100.4, 0) });
    DecodeAndCheck();
}

TEST_F(TSummaryDoubleTsCodecTest, ZeroSum) {
    Encode({ MakePoint("2020-05-30T11:12:00Z", 1, 0, 0, 0, 0) });
    DecodeAndCheck();
}

TEST_F(TSummaryDoubleTsCodecTest, Many) {
    auto ev = TSummaryDoublePoint::EmptyValue;
    Encode({
            MakePoint("2020-05-30T11:12:00Z", ev.CountValue, ev.Sum, ev.Min, ev.Max, ev.Last),
            MakePoint("2020-05-30T11:12:15Z", 1, 5.3, 5.1, 5, 0),
            MakePoint("2020-05-30T11:12:30Z", 2, 6.7, 1.5, 5.2, 0),
            MakePoint("2020-05-30T11:12:45Z", 2, 6.7, 1.5, 5.7, 0),
            MakePoint("2020-05-30T11:13:00Z", 3, 11.23, 1.5, 6.8, 0),
            MakePoint("2020-05-30T11:13:15Z", 161, 241.12, -12.1, 102.12, 0),
    });
    DecodeAndCheck();
}

TEST_F(TSummaryDoubleTsCodecTest, SumOnly) {
    auto ev = TSummaryDoublePoint::EmptyValue;
    Encode({
            MakePoint("2020-05-30T11:12:00Z", ev.CountValue, 1.1, ev.Min, ev.Max, ev.Last),
            MakePoint("2020-05-30T11:12:15Z", ev.CountValue, 2.2, ev.Min, ev.Max, ev.Last),
            MakePoint("2020-05-30T11:12:30Z", ev.CountValue, 3.3, ev.Min, ev.Max, ev.Last),
            MakePoint("2020-05-30T11:12:45Z", ev.CountValue, 4.4, ev.Min, ev.Max, ev.Last),
            MakePoint("2020-05-30T11:13:00Z", ev.CountValue, 5.5, ev.Min, ev.Max, ev.Last),
            MakePoint("2020-05-30T11:13:15Z", ev.CountValue, 6.6, ev.Min, ev.Max, ev.Last),
    });
    DecodeAndCheck();
}

TEST_F(TSummaryDoubleTsCodecTest, MinOnly) {
    auto ev = TSummaryDoublePoint::EmptyValue;
    Encode({
            MakePoint("2020-05-30T11:12:00Z", ev.CountValue, ev.Sum, 1.1, ev.Max, ev.Last),
            MakePoint("2020-05-30T11:12:15Z", ev.CountValue, ev.Sum, 2.2, ev.Max, ev.Last),
            MakePoint("2020-05-30T11:12:30Z", ev.CountValue, ev.Sum, 1.3, ev.Max, ev.Last),
            MakePoint("2020-05-30T11:12:45Z", ev.CountValue, ev.Sum, 2.4, ev.Max, ev.Last),
            MakePoint("2020-05-30T11:13:00Z", ev.CountValue, ev.Sum, 1.5, ev.Max, ev.Last),
            MakePoint("2020-05-30T11:13:15Z", ev.CountValue, ev.Sum, 2.6, ev.Max, ev.Last),
    });
    DecodeAndCheck();
}

TEST_F(TSummaryDoubleTsCodecTest, MaxOnly) {
    auto ev = TSummaryDoublePoint::EmptyValue;
    Encode({
            MakePoint("2020-05-30T11:12:00Z", ev.CountValue, ev.Sum, ev.Min, 1.1, ev.Last),
            MakePoint("2020-05-30T11:12:15Z", ev.CountValue, ev.Sum, ev.Min, 2.2, ev.Last),
            MakePoint("2020-05-30T11:12:30Z", ev.CountValue, ev.Sum, ev.Min, 1.3, ev.Last),
            MakePoint("2020-05-30T11:12:45Z", ev.CountValue, ev.Sum, ev.Min, 2.4, ev.Last),
            MakePoint("2020-05-30T11:13:00Z", ev.CountValue, ev.Sum, ev.Min, 1.5, ev.Last),
            MakePoint("2020-05-30T11:13:15Z", ev.CountValue, ev.Sum, ev.Min, 2.6, ev.Last),
    });
    DecodeAndCheck();
}

TEST_F(TSummaryDoubleTsCodecTest, LastOnly) {
    auto ev = TSummaryDoublePoint::EmptyValue;
    Encode({
            MakePoint("2020-05-30T11:12:00Z", ev.CountValue, ev.Sum, ev.Min, ev.Last, 1.1),
            MakePoint("2020-05-30T11:12:15Z", ev.CountValue, ev.Sum, ev.Min, ev.Last, 2.2),
            MakePoint("2020-05-30T11:12:30Z", ev.CountValue, ev.Sum, ev.Min, ev.Last, 1.3),
            MakePoint("2020-05-30T11:12:45Z", ev.CountValue, ev.Sum, ev.Min, ev.Last, 2.4),
            MakePoint("2020-05-30T11:13:00Z", ev.CountValue, ev.Sum, ev.Min, ev.Last, 1.5),
            MakePoint("2020-05-30T11:13:15Z", ev.CountValue, ev.Sum, ev.Min, ev.Last, 2.6),
    });
    DecodeAndCheck();
}
