#include <solomon/libs/cpp/ts_codec/summary_int_ts_codec.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace NSolomon::NTs;

class TSummaryIntTsCodecTest: public ::testing::Test {
public:
    void SetUp() override {
        Buf_ = {};
    }

    void Encode(std::vector<TSummaryIntPoint> points) {
        Points_ = std::move(points);

        TBitWriter w{&Buf_};
        auto encoder = TSummaryIntTsEncoder::Aggr(&w);
        for (const auto& p: Points_) {
            encoder.EncodePoint(p);
        }

        encoder.Flush();
    }

    void DecodeAndCheck() {
        auto decoder = TSummaryIntTsDecoder::Aggr(Buf_);
        ASSERT_TRUE(decoder.HasNext());

        TSummaryIntPoint point;
        for (size_t i = 0; i < Points_.size(); ++i) {
            const auto& expected = Points_[i];

            bool hasNext;
            ASSERT_NO_THROW({ hasNext = decoder.NextPoint(&point); }) << "point #" << i;
            ASSERT_TRUE(hasNext) << "point #" << i;

            // common fields
            EXPECT_EQ(expected.Time, point.Time) << "point #" << i;
            EXPECT_EQ(expected.Step, point.Step) << "point #" << i;

            // summary fields
            EXPECT_EQ(expected.CountValue, point.CountValue) << "point #" << i;
            EXPECT_EQ(expected.Sum, point.Sum) << "point #" << i;
            EXPECT_EQ(expected.Min, point.Min) << "point #" << i;
            EXPECT_EQ(expected.Max, point.Max) << "point #" << i;
            EXPECT_EQ(expected.Last, point.Last) << "point #" << i;

            // aggregate fields
            EXPECT_EQ(expected.Merge, point.Merge) << "point #" << i;
            EXPECT_EQ(expected.Count, point.Count) << "point #" << i;
        }

        ASSERT_FALSE(decoder.HasNext());
    }

    static TSummaryIntPoint MakePoint(TStringBuf time, i64 count, i64 sum, i64 min, i64 max, i64 last) {
        TSummaryIntPoint point;
        point.Time = TInstant::ParseIso8601(time);
        point.CountValue = count;
        point.Sum = sum;
        point.Min = min;
        point.Max = max;
        point.Last = last;
        return point;
    }

private:
    std::vector<TSummaryIntPoint> Points_;
    TBitBuffer Buf_;
};

TEST_F(TSummaryIntTsCodecTest, One) {
    Encode({ MakePoint("2020-05-30T11:12:00Z", 10, 123, 3, 100, 0) });
    DecodeAndCheck();
}

TEST_F(TSummaryIntTsCodecTest, ZeroSum) {
    Encode({ MakePoint("2020-05-30T11:12:00Z", 1, 0, 0, 0, 0) });
    DecodeAndCheck();
}

TEST_F(TSummaryIntTsCodecTest, Many) {
    auto ev = TSummaryIntPoint::EmptyValue;
    Encode({
            MakePoint("2020-05-30T11:12:00Z", ev.CountValue, ev.Sum, ev.Min, ev.Max, ev.Last),
            MakePoint("2020-05-30T11:12:15Z", 1, 5, 5, 5, 0),
            MakePoint("2020-05-30T11:12:30Z", 2, 6, 1, 5, 0),
            MakePoint("2020-05-30T11:12:45Z", 2, 7, 1, 5, 0),
            MakePoint("2020-05-30T11:13:00Z", 3, 11, 1, 6, 0),
            MakePoint("2020-05-30T11:13:15Z", 161, 241, -12, 102, 0),
    });
    DecodeAndCheck();
}

TEST_F(TSummaryIntTsCodecTest, SumOnly) {
    auto ev = TSummaryIntPoint::EmptyValue;
    Encode({
            MakePoint("2020-05-30T11:12:00Z", ev.CountValue,  1, ev.Min, ev.Max, ev.Last),
            MakePoint("2020-05-30T11:12:15Z", ev.CountValue,  2, ev.Min, ev.Max, ev.Last),
            MakePoint("2020-05-30T11:12:30Z", ev.CountValue,  0, ev.Min, ev.Max, ev.Last),
            MakePoint("2020-05-30T11:12:45Z", ev.CountValue,  3, ev.Min, ev.Max, ev.Last),
            MakePoint("2020-05-30T11:13:00Z", ev.CountValue, -1, ev.Min, ev.Max, ev.Last),
            MakePoint("2020-05-30T11:13:15Z", ev.CountValue,  4, ev.Min, ev.Max, ev.Last),
    });
    DecodeAndCheck();
}

TEST_F(TSummaryIntTsCodecTest, MinOnly) {
    auto ev = TSummaryIntPoint::EmptyValue;
    Encode({
            MakePoint("2020-05-30T11:12:00Z", ev.CountValue, ev.Sum,  1, ev.Max, ev.Last),
            MakePoint("2020-05-30T11:12:15Z", ev.CountValue, ev.Sum,  2, ev.Max, ev.Last),
            MakePoint("2020-05-30T11:12:30Z", ev.CountValue, ev.Sum,  0, ev.Max, ev.Last),
            MakePoint("2020-05-30T11:12:45Z", ev.CountValue, ev.Sum,  3, ev.Max, ev.Last),
            MakePoint("2020-05-30T11:13:00Z", ev.CountValue, ev.Sum, -1, ev.Max, ev.Last),
            MakePoint("2020-05-30T11:13:15Z", ev.CountValue, ev.Sum,  4, ev.Max, ev.Last),
    });
    DecodeAndCheck();
}

TEST_F(TSummaryIntTsCodecTest, MaxOnly) {
    auto ev = TSummaryIntPoint::EmptyValue;
    Encode({
            MakePoint("2020-05-30T11:12:00Z", ev.CountValue, ev.Sum, ev.Min,  1, ev.Last),
            MakePoint("2020-05-30T11:12:15Z", ev.CountValue, ev.Sum, ev.Min,  2, ev.Last),
            MakePoint("2020-05-30T11:12:30Z", ev.CountValue, ev.Sum, ev.Min,  0, ev.Last),
            MakePoint("2020-05-30T11:12:45Z", ev.CountValue, ev.Sum, ev.Min,  3, ev.Last),
            MakePoint("2020-05-30T11:13:00Z", ev.CountValue, ev.Sum, ev.Min, -1, ev.Last),
            MakePoint("2020-05-30T11:13:15Z", ev.CountValue, ev.Sum, ev.Min,  4, ev.Last),
    });
    DecodeAndCheck();
}

TEST_F(TSummaryIntTsCodecTest, LastOnly) {
    auto ev = TSummaryIntPoint::EmptyValue;
    Encode({
            MakePoint("2020-05-30T11:12:00Z", ev.CountValue, ev.Sum, ev.Min, ev.Last,  1),
            MakePoint("2020-05-30T11:12:15Z", ev.CountValue, ev.Sum, ev.Min, ev.Last,  2),
            MakePoint("2020-05-30T11:12:30Z", ev.CountValue, ev.Sum, ev.Min, ev.Last,  0),
            MakePoint("2020-05-30T11:12:45Z", ev.CountValue, ev.Sum, ev.Min, ev.Last,  3),
            MakePoint("2020-05-30T11:13:00Z", ev.CountValue, ev.Sum, ev.Min, ev.Last, -1),
            MakePoint("2020-05-30T11:13:15Z", ev.CountValue, ev.Sum, ev.Min, ev.Last,  4),
    });
    DecodeAndCheck();
}
