#include <solomon/libs/cpp/ts_codec/time_codec.h>
#include <solomon/libs/cpp/ts_codec/ut/helpers.h>

#include <library/cpp/testing/gtest/gtest.h>

#include <util/random/random.h>
#include <util/string/cast.h>

using namespace NSolomon::NTs;

namespace {

std::vector<std::tuple<TStringBuf, bool, i64, TStringBuf>> Data = {
        // time to encode,           millis, delta, next encoded block
        {"2020-05-30T11:12:13.000Z", false,     -1, "00010011 01111100 00010010 10100110 01001110 10000000 00000000 00000000"},
        {"2020-05-30T11:12:20.000Z", false,  7'000, "00000100 01011010 00010010 10100110 01001110 10000000 00000000 00000000"},
        {"2020-05-30T11:12:30.000Z", false, 10'000, "100110"},
        {"2020-05-30T11:12:40.000Z", false, 10'000, "0"},
        {"2020-05-30T11:12:50.123Z",  true, 10'123, "11111111 11001101 111"},
        {"2020-05-30T11:13:00.000Z",  true,  9'877, "11101101 01111000"},
        {"2020-05-30T11:13:10.000Z",  true, 10'000, "11001101 111"},
};

} // namespace

TEST(TTimeCodecTest, InitState) {
    TTimeEncoder encoder;

    EXPECT_EQ(encoder.DeltaMillis(), -1);
    EXPECT_EQ(encoder.Prev(), TInstant::Zero());
    EXPECT_FALSE(encoder.Millis());
}

TEST(TTimeCodecTest, Encode) {
    TTimeEncoder encoder;

    for (const auto& [tsStr, millis, deltaMillis, expectedBits]: Data) {
        TBitBuffer buffer;
        TBitWriter w{&buffer};

        TInstant ts = TInstant::ParseIso8601(tsStr);
        encoder.Encode(&w, ts);

        EXPECT_EQ(encoder.Prev(), ts) << "ts=" << tsStr;
        EXPECT_EQ(encoder.Millis(), millis) << "ts=" << tsStr;
        EXPECT_EQ(encoder.DeltaMillis(), deltaMillis) << "ts=" << tsStr;

        w.Flush();
        ASSERT_TRUE(IsSameBuf(buffer, expectedBits)) << "ts=" << tsStr;
    }
}

TEST(TTimeCodecTest, Decode) {
    TTimeDecoder decoder;

    for (const auto& [tsStr, millis, deltaMillis, bits]: Data) {
        auto [buf, len] = ParseBin2(bits);
        TBitReader r{buf.data(), len};

        TInstant expected = TInstant::ParseIso8601(tsStr);
        TInstant ts;

        ASSERT_NO_THROW({ ts = decoder.Decode(&r); }) << "expected=" << tsStr;
        ASSERT_EQ(ts, expected);
    }
}

TEST(TTimeCodecTest, EncodeDecodeStress) {
    SetRandomSeed(1137);

    TBitBuffer buf;
    TBitWriter w{&buf};
    TTimeEncoder encoder;

    std::vector<TInstant> expected;
    expected.reserve(1000);

    TInstant now = TInstant::ParseIso8601("2020-05-30T11:12:13.000Z");
    for (size_t i = 0; i < 1000; ++i) {
        TInstant ts = now - TDuration::Seconds(RandomNumber(10'000'000u));
        expected.push_back(ts);
        encoder.Encode(&w, ts);
    }
    w.Flush();

    TTimeDecoder decoder;
    TBitReader r{buf};

    for (size_t i = 0; r.Left() != 0; i++) {
        TInstant ts;
        ASSERT_NO_THROW({ ts = decoder.Decode(&r); }) << "expected=" << ToString(expected[i]);
        ASSERT_EQ(ts, expected[i]) << "expected=" << ToString(expected[i]);
    }
}
