#pragma once

#include <solomon/libs/cpp/ts_codec/points.h>

#include <util/generic/yexception.h>

#include <jni.h>

namespace NSolomon::NJava {

class THistogramClass {
public:
    explicit THistogramClass(JNIEnv* jenv) {
        Class_ = jenv->FindClass("ru/yandex/solomon/model/type/Histogram");
        Y_ENSURE(Class_, "cannot find class ru/yandex/solomon/model/type/Histogram");

        NewInstanceMethod_ = jenv->GetStaticMethodID(Class_, "newInstance", "([D[J)Lru/yandex/solomon/model/type/Histogram;");
        Y_ENSURE(NewInstanceMethod_, "cannot find newInstance() method");

        BoundsField_ = GetField(jenv, "bounds", "[D");
        BucketsField_ = GetField(jenv, "buckets", "[J");
        DenomField_ = GetField(jenv, "denom", "J");
        SizeField_ = GetField(jenv, "size", "I");
    }

    jclass Class() const noexcept {
        return Class_;
    }

    jobject ToJava(JNIEnv* jenv, const NTs::NValue::THistogram& hist) {
        std::vector<jdouble> boundData;
        std::vector<jlong> bucketData;
        for (const auto& b: hist.Buckets) {
            boundData.push_back(static_cast<jdouble>(b.UpperBound));
            bucketData.push_back(static_cast<jlong>(b.Value));
        }

        jdoubleArray bounds = jenv->NewDoubleArray(boundData.size());
        jenv->SetDoubleArrayRegion(bounds, 0, boundData.size(), boundData.data());

        jlongArray buckets = jenv->NewLongArray(bucketData.size());
        jenv->SetLongArrayRegion(buckets, 0, bucketData.size(), bucketData.data());

        jobject histJava = jenv->CallStaticObjectMethod(Class_, NewInstanceMethod_, bounds, buckets);
        jenv->SetLongField(histJava, DenomField_, static_cast<jlong>(hist.Denom));

        return histJava;
    }

    NTs::NValue::THistogram ToHist(JNIEnv* jenv, jobject obj) {
        NTs::NValue::THistogram hist;
        hist.Denom = static_cast<ui64>(jenv->GetLongField(obj, DenomField_));

        auto bounds = reinterpret_cast<jdoubleArray>(jenv->GetObjectField(obj, BoundsField_));
        jdouble* boundsData = jenv->GetDoubleArrayElements(bounds, nullptr);

        auto buckets = reinterpret_cast<jlongArray>(jenv->GetObjectField(obj, BucketsField_));
        jlong* bucketsData = jenv->GetLongArrayElements(buckets, nullptr);

        size_t size = static_cast<size_t>(jenv->GetIntField(obj, SizeField_));
        hist.Buckets.reserve(size);
        for (size_t i = 0; i < size; ++i) {
            hist.Buckets.push_back(NTs::NValue::THistogram::TBucket{
                    static_cast<double>(boundsData[i]),
                    static_cast<ui64>(bucketsData[i])
            });
        }
        return hist;
    }

private:
    jfieldID GetField(JNIEnv* jenv, const char* name, const char* sig) {
        jfieldID field = jenv->GetFieldID(Class_, name, sig);
        Y_ENSURE(field, "cannot find field " << name);
        return field;
    }

private:
    jclass Class_;
    jmethodID NewInstanceMethod_;
    jfieldID BoundsField_;
    jfieldID BucketsField_;
    jfieldID DenomField_;
    jfieldID SizeField_;
};

class TLogHistogramClass {
public:
    explicit TLogHistogramClass(JNIEnv* jenv) {
        Class_ = jenv->FindClass("ru/yandex/solomon/model/type/LogHistogram");
        Y_ENSURE(Class_, "cannot find class ru/yandex/solomon/model/type/LogHistogram");

        NewInstanceMethod_ = jenv->GetStaticMethodID(Class_, "newInstance", "()Lru/yandex/solomon/model/type/LogHistogram;");
        Y_ENSURE(NewInstanceMethod_, "cannot find newInstance() method");

        BucketsField_ = GetField(jenv, "buckets", "[D");
        CountZeroField_ = GetField(jenv, "countZero", "J");
        StartPowerField_ = GetField(jenv, "startPower", "I");
        MaxBucketsSizeField_ = GetField(jenv, "maxBucketsSize", "I");
        BaseField_ = GetField(jenv, "base", "D");
        SizeField_ = GetField(jenv, "size", "I");
    }

    jclass Class() const noexcept {
        return Class_;
    }

    jobject ToJava(JNIEnv* jenv, const NTs::NValue::TLogHistogram& hist) {
        jobject histJava = jenv->CallStaticObjectMethod(Class_, NewInstanceMethod_);

        jdoubleArray values = jenv->NewDoubleArray(hist.Values.size());
        jenv->SetDoubleArrayRegion(values, 0, hist.Values.size(), hist.Values.data());
        jenv->SetObjectField(histJava, BucketsField_, values);

        jenv->SetLongField(histJava, CountZeroField_, static_cast<jlong>(hist.ZeroCount));
        jenv->SetIntField(histJava, StartPowerField_, static_cast<jint>(hist.StartPower));
        jenv->SetIntField(histJava, MaxBucketsSizeField_, static_cast<jint>(hist.MaxBucketCount));
        jenv->SetDoubleField(histJava, BaseField_, static_cast<jdouble>(hist.Base));
        jenv->SetIntField(histJava, SizeField_, static_cast<jint>(hist.Values.size()));

        return histJava;
    }

    NTs::NValue::TLogHistogram ToHist(JNIEnv* jenv, jobject obj) {
        size_t size = static_cast<size_t>(jenv->GetIntField(obj, SizeField_));

        auto buckets = reinterpret_cast<jdoubleArray>(jenv->GetObjectField(obj, BucketsField_));
        jdouble* bucketsData = jenv->GetDoubleArrayElements(buckets, nullptr);

        NTs::NValue::TLogHistogram hist;
        hist.Values.reserve(size);
        std::copy(bucketsData, bucketsData + size, std::back_insert_iterator(hist.Values));

        hist.ZeroCount = static_cast<ui64>(jenv->GetLongField(obj, CountZeroField_));
        hist.StartPower = static_cast<i16>(jenv->GetIntField(obj, StartPowerField_));
        hist.MaxBucketCount = static_cast<i16>(jenv->GetIntField(obj, MaxBucketsSizeField_));
        hist.Base = static_cast<double>(jenv->GetDoubleField(obj, BaseField_));
        return hist;
    }

private:
    jfieldID GetField(JNIEnv* jenv, const char* name, const char* sig) {
        jfieldID field = jenv->GetFieldID(Class_, name, sig);
        Y_ENSURE(field, "cannot find field " << name);
        return field;
    }

private:
    jclass Class_;
    jmethodID NewInstanceMethod_;
    jfieldID BucketsField_;
    jfieldID CountZeroField_;
    jfieldID StartPowerField_;
    jfieldID MaxBucketsSizeField_;
    jfieldID BaseField_;
    jfieldID SizeField_;
};

} // namespace NSolomon::NJava
