#pragma once

#include <jni.h>

#include <util/generic/yexception.h>

namespace NSolomon::NJava {

template <typename T>
class TSummaryClass {
    using JavaType = std::conditional_t<std::is_same_v<T, i64>, jlong, jdouble>;

public:
    explicit TSummaryClass(JNIEnv* jenv) {
        const char* className;
        const char* constructorSignature;
        const char* fieldType;

        if constexpr (std::is_same_v<T, i64>) {
            className = "ru/yandex/monlib/metrics/summary/ImmutableSummaryInt64Snapshot";
            constructorSignature = "(JJJJJ)V";
            fieldType = "J";
            Getter_ = &JNIEnv::GetLongField;
        } else if constexpr (std::is_same_v<T, double>) {
            className = "ru/yandex/monlib/metrics/summary/ImmutableSummaryDoubleSnapshot";
            constructorSignature = "(JDDDD)V";
            fieldType = "D";
            Getter_ = &JNIEnv::GetDoubleField;
        } else {
            static_assert(TDependentFalse<T>, "unknown type");
        }

        Class_ = jenv->FindClass(className);
        Y_ENSURE(Class_, "cannot find class " << className);

        Constructor_ = jenv->GetMethodID(Class_, "<init>", constructorSignature);
        Y_ENSURE(Constructor_, "cannot find constructor");

        CountField_ = GetField(jenv, "count", "J");
        SumField_ = GetField(jenv, "sum", fieldType);
        MinField_ = GetField(jenv, "min", fieldType);
        MaxField_ = GetField(jenv, "max", fieldType);
        LastField_ = GetField(jenv, "last", fieldType);
    }

    jclass Class() const noexcept {
        return Class_;
    }

    jobject New(JNIEnv* jenv, i64 count, T sum, T min, T max, T last) {
        return jenv->NewObject(
                Class_,
                Constructor_,
                static_cast<jlong>(count),
                static_cast<JavaType>(sum),
                static_cast<JavaType>(min),
                static_cast<JavaType>(max),
                static_cast<JavaType>(last));
    }

    i64 GetCount(JNIEnv* jenv, jobject obj) {
        return static_cast<i64>(jenv->GetLongField(obj, CountField_));
    }

    T GetSum(JNIEnv* jenv, jobject obj) {
        return static_cast<T>(std::invoke(Getter_, jenv, obj, SumField_));
    }

    T GetMin(JNIEnv* jenv, jobject obj) {
        return static_cast<T>(std::invoke(Getter_, jenv, obj, MinField_));
    }

    T GetMax(JNIEnv* jenv, jobject obj) {
        return static_cast<T>(std::invoke(Getter_, jenv, obj, MaxField_));
    }

    T GetLast(JNIEnv* jenv, jobject obj) {
        return static_cast<T>(std::invoke(Getter_, jenv, obj, LastField_));
    }

private:
    jfieldID GetField(JNIEnv* jenv, const char* name, const char* sig) {
        jfieldID field = jenv->GetFieldID(Class_, name, sig);
        Y_ENSURE(field, "cannot find field " << name);
        return field;
    }

private:
    jclass Class_;
    jmethodID Constructor_;
    jfieldID CountField_;
    jfieldID SumField_;
    jfieldID MinField_;
    jfieldID MaxField_;
    jfieldID LastField_;

    JavaType (JNIEnv::*Getter_)(jobject obj, jfieldID fieldID);
};

using TSummaryInt64Class = TSummaryClass<i64>;
using TSummaryDoubleClass = TSummaryClass<double>;

} // namespace NSolomon::NJava
