package ru.yandex.solomon.ts_codec;

import java.util.List;
import java.util.Random;

import it.unimi.dsi.fastutil.booleans.BooleanArrayList;
import it.unimi.dsi.fastutil.bytes.ByteArrayList;
import it.unimi.dsi.fastutil.doubles.DoubleArrayList;
import it.unimi.dsi.fastutil.ints.IntArrayList;
import it.unimi.dsi.fastutil.longs.LongArrayList;
import org.junit.Assert;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.solomon.codec.bits.HeapBitBuf;

@RunWith(Parameterized.class)
public class BitReaderTest {

    private static final int COUNT = 100;
    private static final Random rnd = new Random(71);

    @Parameterized.Parameters(name = "offset: {0}")
    public static List<Integer> offsets() {
        return List.of(0, 3, 5, 7);
    }

    @Parameterized.Parameter
    public int offset;

    @Test
    public void readBits() {
        HeapBitBuf bitBuf = new HeapBitBuf();
        fillOffset(bitBuf, offset);

        BooleanArrayList expected = new BooleanArrayList();
        for (int i = 0; i < COUNT; i++) {
            boolean bit = rnd.nextBoolean();
            expected.add(bit);
            bitBuf.writeBit(bit);
        }

        boolean[] bits = BitStreamNative.readBits(bitBuf, offset);
        Assert.assertArrayEquals(expected.toBooleanArray(), bits);
    }

    @Test
    public void readInt8() {
        HeapBitBuf bitBuf = new HeapBitBuf();
        fillOffset(bitBuf, offset);

        ByteArrayList expected = new ByteArrayList();
        for (int i = 0; i < COUNT; i++) {
            byte value = (byte) rnd.nextInt(256);
            expected.add(value);
            bitBuf.write8Bits(value);
        }

        byte[] values = BitStreamNative.readInt8(bitBuf, offset);
        Assert.assertArrayEquals(expected.toByteArray(), values);
    }

    @Test
    public void readInt32() {
        HeapBitBuf bitBuf = new HeapBitBuf();
        fillOffset(bitBuf, offset);

        IntArrayList expected = new IntArrayList();
        for (int i = 0; i < COUNT; i++) {
            int value = rnd.nextInt();
            expected.add(value);
            bitBuf.write32Bits(value);
        }

        int[] values = BitStreamNative.readInt32(bitBuf, offset);
        Assert.assertArrayEquals(expected.toIntArray(), values);
    }

    @Test
    public void readInt64() {
        HeapBitBuf bitBuf = new HeapBitBuf();
        fillOffset(bitBuf, offset);

        LongArrayList expected = new LongArrayList();
        for (int i = 0; i < COUNT; i++) {
            long value = rnd.nextLong();
            expected.add(value);
            bitBuf.write64Bits(value);
        }

        long[] values = BitStreamNative.readInt64(bitBuf, offset);
        Assert.assertArrayEquals(expected.toLongArray(), values);
    }

    @Test
    public void readDouble() {
        HeapBitBuf bitBuf = new HeapBitBuf();
        fillOffset(bitBuf, offset);

        DoubleArrayList expected = new DoubleArrayList();
        for (int i = 0; i < COUNT; i++) {
            double value = rnd.nextDouble();
            expected.add(value);
            bitBuf.writeDoubleBits(value);
        }

        double[] values = BitStreamNative.readDouble(bitBuf, offset);
        Assert.assertArrayEquals(expected.toDoubleArray(), values, Double.MIN_NORMAL);
    }

    @Test
    public void readInt32Bits() {
        for (int bits = 1; bits < 32; bits++) {
            HeapBitBuf bitBuf = new HeapBitBuf();
            fillOffset(bitBuf, offset);

            IntArrayList expected = new IntArrayList();
            for (int i = 0; i < COUNT; i++) {
                int value = rnd.nextInt() & ~(0xffffffff << bits); // keep lower bits
                expected.add(value);
                bitBuf.writeBits(value, bits);
            }

            int[] values = BitStreamNative.readInt32Bits(bitBuf, bits, offset);
            Assert.assertArrayEquals("bits: " + bits, expected.toIntArray(), values);
        }
    }

    @Test
    public void readInt64Bits() {
        for (int bits = 1; bits < 64; bits++) {
            HeapBitBuf bitBuf = new HeapBitBuf();
            fillOffset(bitBuf, offset);

            LongArrayList expected = new LongArrayList();
            for (int i = 0; i < COUNT; i++) {
                long value = rnd.nextLong() & ~(0xffffffff_ffffffffL << bits); // keep lower bits
                expected.add(value);
                bitBuf.writeBits(value, bits);
            }

            long[] values = BitStreamNative.readInt64Bits(bitBuf, bits, offset);
            Assert.assertArrayEquals("bits: " + bits, expected.toLongArray(), values);
        }
    }

    @Test
    public void readVarInt32() {
        HeapBitBuf bitBuf = new HeapBitBuf();
        fillOffset(bitBuf, offset);

        IntArrayList expected = new IntArrayList();
        for (int i = 0; i < COUNT; i++) {
            int value = rnd.nextInt();
            expected.add(value);
            bitBuf.writeIntVarint8(value);
        }

        int[] values = BitStreamNative.readVarInt32(bitBuf, offset);
        Assert.assertArrayEquals(expected.toIntArray(), values);
    }

    @Test
    public void readVarInt64() {
        HeapBitBuf bitBuf = new HeapBitBuf();
        fillOffset(bitBuf, offset);

        LongArrayList expected = new LongArrayList();
        for (int i = 0; i < COUNT; i++) {
            long value = rnd.nextLong();
            expected.add(value);
            bitBuf.writeLongVarint8(value);
        }

        long[] values = BitStreamNative.readVarInt64(bitBuf, offset);
        Assert.assertArrayEquals(expected.toLongArray(), values);
    }

    @Test
    public void readOnes() {
        HeapBitBuf bitBuf = new HeapBitBuf();
        fillOffset(bitBuf, offset);

        IntArrayList expected = new IntArrayList();
        for (int i = 0; i < COUNT; i++) {
            int value = rnd.nextInt(9);
            expected.add(value);
            bitBuf.writeIntVarint1N(value, 8);
        }

        int[] values = BitStreamNative.readOnes(bitBuf, offset);
        Assert.assertArrayEquals(expected.toIntArray(), values);
    }

    private static void fillOffset(HeapBitBuf bitBuf, int offset) {
        for (int i = 0; i < offset; i++) {
            bitBuf.writeBit(i % 2 == 0);
        }
    }
}
