package ru.yandex.solomon.ts_codec;

import java.util.List;
import java.util.Random;

import org.junit.Assert;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import ru.yandex.solomon.codec.bits.HeapBitBuf;

@RunWith(Parameterized.class)
public class BitWriterTest {

    private static final int COUNT = 100;
    private static final Random rnd = new Random(37);

    @Parameterized.Parameters(name = "offset: {0}")
    public static List<Integer> offsets() {
        return List.of(0, 3, 5, 7);
    }

    @Parameterized.Parameter
    public int offset;

    @Test
    public void writeBits() {
        boolean[] expected = new boolean[COUNT];
        for (int i = 0; i < expected.length; i++) {
            expected[i] = rnd.nextBoolean();
        }

        HeapBitBuf bitBuf = BitStreamNative.writeBits(expected, offset);
        bitBuf.readerIndex(offset);

        for (boolean e : expected) {
            Assert.assertEquals(e, bitBuf.readBit());
        }
        Assert.assertEquals(0, bitBuf.readableBits());
    }

    @Test
    public void writeInt8() {
        byte[] expected = new byte[COUNT];
        for (int i = 0; i < expected.length; i++) {
            expected[i] = (byte) rnd.nextInt(256);
        }

        HeapBitBuf bitBuf = BitStreamNative.writeInt8(expected, offset);
        bitBuf.readerIndex(offset);

        for (byte e : expected) {
            Assert.assertEquals(e, bitBuf.read8Bits());
        }
        Assert.assertEquals(0, bitBuf.readableBits());
    }

    @Test
    public void writeInt32() {
        int[] expected = new int[COUNT];
        for (int i = 0; i < expected.length; i++) {
            expected[i] = rnd.nextInt();
        }

        HeapBitBuf bitBuf = BitStreamNative.writeInt32(expected, offset);
        bitBuf.readerIndex(offset);

        for (int e : expected) {
            Assert.assertEquals(e, bitBuf.read32Bits());
        }
        Assert.assertEquals(0, bitBuf.readableBits());
    }

    @Test
    public void writeInt64() {
        long[] expected = new long[COUNT];
        for (int i = 0; i < expected.length; i++) {
            expected[i] = rnd.nextLong();
        }

        HeapBitBuf bitBuf = BitStreamNative.writeInt64(expected, offset);
        bitBuf.readerIndex(offset);

        for (long e : expected) {
            Assert.assertEquals(e, bitBuf.read64Bits());
        }
        Assert.assertEquals(0, bitBuf.readableBits());
    }

    @Test
    public void writeDouble() {
        double[] expected = new double[COUNT];
        for (int i = 0; i < expected.length; i++) {
            expected[i] = rnd.nextDouble();
        }

        HeapBitBuf bitBuf = BitStreamNative.writeDouble(expected, offset);
        bitBuf.readerIndex(offset);

        for (double e : expected) {
            Assert.assertEquals(e, bitBuf.readDoubleBits(), Double.MIN_NORMAL);
        }
        Assert.assertEquals(0, bitBuf.readableBits());
    }

    @Test
    public void writeInt32Bits() {
        for (int bits = 1; bits < 32; bits++) {
            int[] expected = new int[COUNT];
            for (int i = 0; i < expected.length; i++) {
                expected[i] = rnd.nextInt() & ~(0xffffffff << bits); // keep lower bits
            }

            HeapBitBuf bitBuf = BitStreamNative.writeInt32Bits(expected, bits, offset);
            bitBuf.readerIndex(offset);

            for (int e : expected) {
                Assert.assertEquals(e, bitBuf.readBitsToInt(bits));
            }
            Assert.assertEquals(0, bitBuf.readableBits());
        }
    }

    @Test
    public void writeInt64Bits() {
        for (int bits = 1; bits < 64; bits++) {
            long[] expected = new long[COUNT];
            for (int i = 0; i < expected.length; i++) {
                expected[i] = rnd.nextLong() & ~(0xffffffff_ffffffffL << bits); // keep lower bits
            }

            HeapBitBuf bitBuf = BitStreamNative.writeInt64Bits(expected, bits, offset);
            bitBuf.readerIndex(offset);

            for (long e : expected) {
                Assert.assertEquals(e, bitBuf.readBitsToLong(bits));
            }
            Assert.assertEquals(0, bitBuf.readableBits());
        }
    }

    @Test
    public void writeVarInt32() {
        int[] expected = new int[COUNT];
        for (int i = 0; i < expected.length; i++) {
            expected[i] = rnd.nextInt();
        }

        HeapBitBuf bitBuf = BitStreamNative.writeVarInt32(expected, offset);
        bitBuf.readerIndex(offset);

        for (int e : expected) {
            Assert.assertEquals(e, bitBuf.readIntVarint8());
        }
        Assert.assertEquals(0, bitBuf.readableBits());
    }

    @Test
    public void writeVarInt64() {
        long[] expected = new long[COUNT];
        for (int i = 0; i < expected.length; i++) {
            expected[i] = rnd.nextLong();
        }

        HeapBitBuf bitBuf = BitStreamNative.writeVarInt64(expected, offset);
        bitBuf.readerIndex(offset);

        for (long e : expected) {
            Assert.assertEquals(e, bitBuf.readLongVarint8());
        }
        Assert.assertEquals(0, bitBuf.readableBits());
    }

    @Test
    public void writeOnes() {
        int[] expected = new int[COUNT];
        for (int i = 0; i < expected.length; i++) {
            expected[i] = rnd.nextInt(9);
        }

        HeapBitBuf bitBuf = BitStreamNative.writeOnes(expected, offset);
        bitBuf.readerIndex(offset);

        for (int e : expected) {
            Assert.assertEquals(e, bitBuf.readIntVarint1N(8));
        }
        Assert.assertEquals(0, bitBuf.readableBits());
    }
}
