package ru.yandex.solomon.ts_codec;

import org.junit.Assert;
import org.junit.Test;

import ru.yandex.solomon.codec.bits.HeapBitBuf;
import ru.yandex.solomon.codec.compress.doubles.DoubleTimeSeriesOutputStream;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.AggrPointDataTestSupport;
import ru.yandex.solomon.util.time.InstantUtils;

import static ru.yandex.solomon.model.point.AggrPoint.fullForTest;
import static ru.yandex.solomon.model.point.AggrPoint.shortPoint;
import static ru.yandex.solomon.model.point.column.StockpileColumn.COUNT;
import static ru.yandex.solomon.model.point.column.StockpileColumn.MERGE;
import static ru.yandex.solomon.model.point.column.StockpileColumn.STEP;
import static ru.yandex.solomon.model.point.column.StockpileColumn.TS;
import static ru.yandex.solomon.model.point.column.StockpileColumn.VALUE;

/**
 * @author Sergey Polovko
 */
public class DoubleTsDecoderTest {

    @Test
    public void simplePoints() {
        final int columnsMask = TS.mask() | VALUE.mask();

        AggrPoint[] expected = {
                shortPoint(ts("2020-05-30T11:12:00Z"), -Math.PI),
                shortPoint(ts("2020-05-30T11:12:15Z"), -Math.E),
                shortPoint(ts("2020-05-30T11:12:30Z"), 0.0),
                shortPoint(ts("2020-05-30T11:12:45Z"), Math.E),
                shortPoint(ts("2020-05-30T11:13:00Z"), Math.PI),
        };

        HeapBitBuf buf = encode(columnsMask, expected);
        AggrPoint[] decoded = TsStreamNative.decodeDouble(columnsMask, buf);
        Assert.assertArrayEquals(expected, decoded);
    }

    @Test
    public void simplePointsWithStepMillis() {
        final int columnsMask = TS.mask() | VALUE.mask() | STEP.mask();

        AggrPoint[] expected = {
                shortPoint(ts("2020-05-30T11:12:00Z"), -Math.PI, 15_000),
                shortPoint(ts("2020-05-30T11:12:15Z"),  -Math.E, 30_000),
                shortPoint(ts("2020-05-30T11:12:30Z"),      0.0, 10_000),
                shortPoint(ts("2020-05-30T11:12:45Z"),   Math.E, 15_000),
                shortPoint(ts("2020-05-30T11:13:00Z"),  Math.PI, 20_000),
        };

        HeapBitBuf buf = encode(columnsMask, expected);
        AggrPoint[] decoded = TsStreamNative.decodeDouble(columnsMask, buf);
        Assert.assertArrayEquals(expected, decoded);
    }

    @Test
    public void aggrPoints() {
        final int columnsMask = TS.mask() | VALUE.mask() | MERGE.mask() | COUNT.mask();

        AggrPoint[] expected = {
                fullForTest(ts("2020-05-30T11:12:00Z"), -Math.PI,  true,  1),
                fullForTest(ts("2020-05-30T11:12:15Z"),  -Math.E, false,  5),
                fullForTest(ts("2020-05-30T11:12:30Z"),      0.0, false,  3),
                fullForTest(ts("2020-05-30T11:12:45Z"),   Math.E,  true,  7),
                fullForTest(ts("2020-05-30T11:13:00Z"),  Math.PI,  true, 10),
        };

        HeapBitBuf buf = encode(columnsMask, expected);
        AggrPoint[] decoded = TsStreamNative.decodeDouble(columnsMask, buf);
        Assert.assertArrayEquals(expected, decoded);
    }

    @Test
    public void stress() {
        final int columnsMask = TS.mask() | VALUE.mask() | STEP.mask() | MERGE.mask() | COUNT.mask();

        AggrPoint[] expected = new AggrPoint[1000];
        for (int i = 0; i < expected.length; i++) {
            expected[i] = AggrPointDataTestSupport.randomPoint(columnsMask);
        }

        HeapBitBuf buf = encode(columnsMask, expected);
        AggrPoint[] decoded = TsStreamNative.decodeDouble(columnsMask, buf);
        Assert.assertArrayEquals(expected, decoded);
    }

    private static HeapBitBuf encode(int columnsMask, AggrPoint[] points) {
        var buf = new HeapBitBuf();
        try (var out = new DoubleTimeSeriesOutputStream(buf, points.length)) {
            for (AggrPoint point : points) {
                out.writePoint(columnsMask, point);
            }
        }
        return buf;
    }

    private static long ts(String str) {
        return InstantUtils.parseToMillis(str);
    }
}
