package ru.yandex.solomon.ts_codec;

import org.junit.Assert;
import org.junit.Test;

import ru.yandex.solomon.codec.bits.HeapBitBuf;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.AggrPointDataTestSupport;

import static ru.yandex.solomon.model.point.AggrPoints.lpoint;
import static ru.yandex.solomon.model.point.column.StockpileColumn.COUNT;
import static ru.yandex.solomon.model.point.column.StockpileColumn.LONG_VALUE;
import static ru.yandex.solomon.model.point.column.StockpileColumn.MERGE;
import static ru.yandex.solomon.model.point.column.StockpileColumn.STEP;
import static ru.yandex.solomon.model.point.column.StockpileColumn.TS;
import static ru.yandex.solomon.model.protobuf.MetricType.IGAUGE;
import static ru.yandex.solomon.ts_codec.TsStreamJava.encode;

/**
 * @author Sergey Polovko
 */
public class GaugeInt64TsDecoderTest {

    @Test
    public void simplePoints() {
        final int columnsMask = TS.mask() | LONG_VALUE.mask();

        AggrPoint[] expected = {
                lpoint("2020-05-30T11:12:00Z", Long.MIN_VALUE),
                lpoint("2020-05-30T11:12:15Z",           -100),
                lpoint("2020-05-30T11:12:30Z",              0),
                lpoint("2020-05-30T11:12:45Z",            100),
                lpoint("2020-05-30T11:13:00Z", Long.MAX_VALUE),
        };

        HeapBitBuf buf = encode(IGAUGE, columnsMask, expected);
        AggrPoint[] decoded = TsStreamNative.gaugeInt64Decode(columnsMask, buf);
        Assert.assertArrayEquals(expected, decoded);
    }

    @Test
    public void simplePointsWithStepMillis() {
        final int columnsMask = TS.mask() | LONG_VALUE.mask() | STEP.mask();

        AggrPoint[] expected = {
                lpoint("2020-05-30T11:12:00Z", Long.MIN_VALUE, 15_000),
                lpoint("2020-05-30T11:12:15Z",           -100, 30_000),
                lpoint("2020-05-30T11:12:30Z",              0, 10_000),
                lpoint("2020-05-30T11:12:45Z",            100, 15_000),
                lpoint("2020-05-30T11:13:00Z", Long.MAX_VALUE, 20_000),
        };

        HeapBitBuf buf = encode(IGAUGE, columnsMask, expected);
        AggrPoint[] decoded = TsStreamNative.gaugeInt64Decode(columnsMask, buf);
        Assert.assertArrayEquals(expected, decoded);
    }

    @Test
    public void aggrPoints() {
        final int columnsMask = TS.mask() | LONG_VALUE.mask() | MERGE.mask() | COUNT.mask();

        AggrPoint[] expected = {
                lpoint("2020-05-30T11:12:00Z", Long.MIN_VALUE,  true,  1),
                lpoint("2020-05-30T11:12:15Z",           -100, false,  5),
                lpoint("2020-05-30T11:12:30Z",              0, false,  3),
                lpoint("2020-05-30T11:12:45Z",            100,  true,  7),
                lpoint("2020-05-30T11:13:00Z", Long.MAX_VALUE,  true, 10),
        };

        HeapBitBuf buf = encode(IGAUGE, columnsMask, expected);
        AggrPoint[] decoded = TsStreamNative.gaugeInt64Decode(columnsMask, buf);
        Assert.assertArrayEquals(expected, decoded);
    }

    @Test
    public void stress() {
        final int columnsMask = TS.mask() | LONG_VALUE.mask() | STEP.mask() | MERGE.mask() | COUNT.mask();

        AggrPoint[] expected = new AggrPoint[1000];
        for (int i = 0; i < expected.length; i++) {
            expected[i] = AggrPointDataTestSupport.randomPoint(columnsMask);
        }

        HeapBitBuf buf = encode(IGAUGE, columnsMask, expected);
        AggrPoint[] decoded = TsStreamNative.gaugeInt64Decode(columnsMask, buf);
        Assert.assertArrayEquals(expected, decoded);
    }
}
