package ru.yandex.solomon.ts_codec;

import org.junit.Assert;
import org.junit.Test;

import ru.yandex.solomon.codec.bits.HeapBitBuf;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.AggrPointDataTestSupport;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.type.Histogram;

import static ru.yandex.solomon.model.point.AggrPoints.point;
import static ru.yandex.solomon.model.point.AggrPoints.pointWithStepMillis;
import static ru.yandex.solomon.model.point.column.StockpileColumn.COUNT;
import static ru.yandex.solomon.model.point.column.StockpileColumn.HISTOGRAM;
import static ru.yandex.solomon.model.point.column.StockpileColumn.MERGE;
import static ru.yandex.solomon.model.point.column.StockpileColumn.STEP;
import static ru.yandex.solomon.model.point.column.StockpileColumn.TS;

/**
 * @author Sergey Polovko
 */
public class HistogramTsDecoderTest {

    @Test
    public void simplePoints() {
        final int columnsMask = TS.mask() | HISTOGRAM.mask();

        AggrPoint[] expected = {
                point("2020-05-30T11:12:00Z", histogram(new double[]{10, 20, 30}, new long[]{0, 3, 2})),
                point("2020-05-30T11:12:15Z", histogram(new double[]{10, 20, 30}, new long[]{0, 3, 2})),
                point("2020-05-30T11:12:30Z", histogram(new double[]{10, 20, 30}, new long[]{4, 3, 8})),
                point("2020-05-30T11:12:45Z", histogram(new double[]{10, 20, 30}, new long[]{5, 9, 8})),
                point("2020-05-30T11:13:00Z", histogram(new double[]{10, 20, 30}, new long[]{9, 9, 9})),
        };

        HeapBitBuf buf = TsStreamJava.encode(MetricType.HIST, columnsMask, expected);
        AggrPoint[] decoded = TsStreamNative.histogramDecode(columnsMask, buf);
        Assert.assertArrayEquals(expected, decoded);
    }

    @Test
    public void simplePointsWithStepMillis() {
        final int columnsMask = TS.mask() | HISTOGRAM.mask() | STEP.mask();

        AggrPoint[] expected = {
                pointWithStepMillis("2020-05-30T11:12:00Z", histogram(new double[]{10, 20, 30}, new long[]{0, 3, 2}), 15_000),
                pointWithStepMillis("2020-05-30T11:12:15Z", histogram(new double[]{10, 20, 30}, new long[]{0, 3, 2}), 30_000),
                pointWithStepMillis("2020-05-30T11:12:30Z", histogram(new double[]{10, 20, 30}, new long[]{4, 3, 8}), 10_000),
                pointWithStepMillis("2020-05-30T11:12:45Z", histogram(new double[]{10, 20, 30}, new long[]{5, 9, 8}), 15_000),
                pointWithStepMillis("2020-05-30T11:13:00Z", histogram(new double[]{10, 20, 30}, new long[]{9, 9, 9}), 20_000),
        };

        HeapBitBuf buf = TsStreamJava.encode(MetricType.HIST, columnsMask, expected);
        AggrPoint[] decoded = TsStreamNative.histogramDecode(columnsMask, buf);
        Assert.assertArrayEquals(expected, decoded);
    }

    @Test
    public void aggrPoints() {
        final int columnsMask = TS.mask() | HISTOGRAM.mask() | MERGE.mask() | COUNT.mask();

        AggrPoint[] expected = {
                point("2020-05-30T11:12:00Z", histogram(new double[]{10, 20, 30}, new long[]{0, 3, 2}),  true,  1),
                point("2020-05-30T11:12:15Z", histogram(new double[]{10, 20, 30}, new long[]{0, 3, 2}), false,  5),
                point("2020-05-30T11:12:30Z", histogram(new double[]{10, 20, 30}, new long[]{4, 3, 8}), false,  3),
                point("2020-05-30T11:12:45Z", histogram(new double[]{10, 20, 30}, new long[]{5, 9, 8}),  true,  7),
                point("2020-05-30T11:13:00Z", histogram(new double[]{10, 20, 30}, new long[]{9, 9, 9}),  true, 10),
        };

        HeapBitBuf buf = TsStreamJava.encode(MetricType.HIST, columnsMask, expected);
        AggrPoint[] decoded = TsStreamNative.histogramDecode(columnsMask, buf);
        Assert.assertArrayEquals(expected, decoded);
    }

    @Test
    public void emptyHist() {
        final int columnsMask = TS.mask() | HISTOGRAM.mask() | MERGE.mask() | COUNT.mask();

        AggrPoint[] expected = {
                point("2020-05-30T11:12:00Z", histogram(new double[]{10, 20, 30}, new long[]{0, 3, 2}),  true,  1),
                point("2020-05-30T11:12:15Z", histogram(new double[0], new long[0]), false,  5)
        };

        HeapBitBuf buf = TsStreamJava.encode(MetricType.HIST, columnsMask, expected);
        AggrPoint[] decoded = TsStreamNative.histogramDecode(columnsMask, buf);
        Assert.assertArrayEquals(expected, decoded);
    }

    @Test
    public void stress() {
        final int columnsMask = TS.mask() | HISTOGRAM.mask() | STEP.mask() | MERGE.mask() | COUNT.mask();

        AggrPoint[] expected = new AggrPoint[1000];
        for (int i = 0; i < expected.length; i++) {
            expected[i] = AggrPointDataTestSupport.randomPoint(columnsMask);
        }

        HeapBitBuf buf = TsStreamJava.encode(MetricType.HIST, columnsMask, expected);
        AggrPoint[] decoded = TsStreamNative.histogramDecode(columnsMask, buf);
        Assert.assertArrayEquals(expected, decoded);
    }

    private static Histogram histogram(double[] bounds, long[] buckets) {
        return Histogram.newInstance(bounds, buckets);
    }
}
