package ru.yandex.solomon.ts_codec;

import org.junit.Assert;
import org.junit.Test;

import ru.yandex.solomon.codec.bits.HeapBitBuf;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.AggrPointDataTestSupport;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.type.LogHistogram;

import static ru.yandex.solomon.model.point.AggrPoints.point;
import static ru.yandex.solomon.model.point.AggrPoints.pointWithStepMillis;
import static ru.yandex.solomon.model.point.column.StockpileColumn.COUNT;
import static ru.yandex.solomon.model.point.column.StockpileColumn.LOG_HISTOGRAM;
import static ru.yandex.solomon.model.point.column.StockpileColumn.MERGE;
import static ru.yandex.solomon.model.point.column.StockpileColumn.STEP;
import static ru.yandex.solomon.model.point.column.StockpileColumn.TS;

/**
 * @author Sergey Polovko
 */
public class LogHistogramTsDecoderTest {

    @Test
    public void simplePoints() {
        final int columnsMask = TS.mask() | LOG_HISTOGRAM.mask();

        AggrPoint[] expected = {
                point("2020-05-30T11:12:00Z", histogram(1.5, 1, 10, 15, new double[]{1.1, 3.1, 2.1})),
                point("2020-05-30T11:12:15Z", histogram(1.5, 2,  9, 10, new double[]{2.1, 4.1, 3.1})),
                point("2020-05-30T11:12:30Z", histogram(1.5, 3, 11, 20, new double[]{3.1, 5.1, 4.1})),
                point("2020-05-30T11:12:45Z", histogram(2.5, 4,  8, 15, new double[]{4.1, 6.1, 5.1})),
                point("2020-05-30T11:13:00Z", histogram(2.5, 5, 12, 25, new double[]{5.1, 7.1, 6.1})),
        };

        HeapBitBuf buf = TsStreamJava.encode(MetricType.LOG_HISTOGRAM, columnsMask, expected);
        AggrPoint[] decoded = TsStreamNative.logHistogramDecode(columnsMask, buf);
        Assert.assertArrayEquals(expected, decoded);
    }

    @Test
    public void simplePointsWithStepMillis() {
        final int columnsMask = TS.mask() | LOG_HISTOGRAM.mask() | STEP.mask();

        AggrPoint[] expected = {
                pointWithStepMillis("2020-05-30T11:12:00Z", histogram(1.5, 1, 10, 15, new double[]{1.1, 3.1, 2.1}), 15_000),
                pointWithStepMillis("2020-05-30T11:12:15Z", histogram(1.5, 2,  9, 10, new double[]{2.1, 4.1, 3.1}), 30_000),
                pointWithStepMillis("2020-05-30T11:12:30Z", histogram(1.5, 3, 11, 20, new double[]{3.1, 5.1, 4.1}), 10_000),
                pointWithStepMillis("2020-05-30T11:12:45Z", histogram(2.5, 4,  8, 15, new double[]{4.1, 6.1, 5.1}), 15_000),
                pointWithStepMillis("2020-05-30T11:13:00Z", histogram(2.5, 5, 12, 25, new double[]{5.1, 7.1, 6.1}), 20_000),
        };

        HeapBitBuf buf = TsStreamJava.encode(MetricType.LOG_HISTOGRAM, columnsMask, expected);
        AggrPoint[] decoded = TsStreamNative.logHistogramDecode(columnsMask, buf);
        Assert.assertArrayEquals(expected, decoded);
    }

    @Test
    public void aggrPoints() {
        final int columnsMask = TS.mask() | LOG_HISTOGRAM.mask() | MERGE.mask() | COUNT.mask();

        AggrPoint[] expected = {
                point("2020-05-30T11:12:00Z", histogram(1.5, 1, 10, 15, new double[]{1.1, 3.1, 2.1}),  true,  1),
                point("2020-05-30T11:12:15Z", histogram(1.5, 2,  9, 10, new double[]{2.1, 4.1, 3.1}), false,  5),
                point("2020-05-30T11:12:30Z", histogram(1.5, 3, 11, 20, new double[]{3.1, 5.1, 4.1}), false,  3),
                point("2020-05-30T11:12:45Z", histogram(2.5, 4,  8, 15, new double[]{4.1, 6.1, 5.1}),  true,  7),
                point("2020-05-30T11:13:00Z", histogram(2.5, 5, 12, 25, new double[]{5.1, 7.1, 6.1}),  true, 10),
        };

        HeapBitBuf buf = TsStreamJava.encode(MetricType.LOG_HISTOGRAM, columnsMask, expected);
        AggrPoint[] decoded = TsStreamNative.logHistogramDecode(columnsMask, buf);
        Assert.assertArrayEquals(expected, decoded);
    }

    @Test
    public void stress() {
        final int columnsMask = TS.mask() | LOG_HISTOGRAM.mask() | STEP.mask() | MERGE.mask() | COUNT.mask();

        AggrPoint[] expected = new AggrPoint[1000];
        for (int i = 0; i < expected.length; i++) {
            expected[i] = AggrPointDataTestSupport.randomPoint(columnsMask);
        }

        HeapBitBuf buf = TsStreamJava.encode(MetricType.LOG_HISTOGRAM, columnsMask, expected);
        AggrPoint[] decoded = TsStreamNative.logHistogramDecode(columnsMask, buf);
        Assert.assertArrayEquals(expected, decoded);
    }

    private static LogHistogram histogram(double base, int startPower, long countZero, int maxBucketsSize, double[] buckets) {
        return LogHistogram.newInstance()
                .setBase(base)
                .setStartPower(startPower)
                .setCountZero(countZero)
                .setMaxBucketsSize(maxBucketsSize)
                .setBuckets(buckets);
    }
}
