package ru.yandex.solomon.ts_codec;

import org.junit.Assert;
import org.junit.Test;

import ru.yandex.monlib.metrics.summary.ImmutableSummaryDoubleSnapshot;
import ru.yandex.solomon.codec.bits.HeapBitBuf;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.AggrPointDataTestSupport;
import ru.yandex.solomon.model.protobuf.MetricType;

import static ru.yandex.solomon.model.point.AggrPoints.point;
import static ru.yandex.solomon.model.point.AggrPoints.pointWithStepMillis;
import static ru.yandex.solomon.model.point.column.StockpileColumn.COUNT;
import static ru.yandex.solomon.model.point.column.StockpileColumn.DSUMMARY;
import static ru.yandex.solomon.model.point.column.StockpileColumn.MERGE;
import static ru.yandex.solomon.model.point.column.StockpileColumn.STEP;
import static ru.yandex.solomon.model.point.column.StockpileColumn.TS;

/**
 * @author Sergey Polovko
 */
public class SummaryDoubleTsEncoderTest {

    @Test
    public void simplePoints() {
        final int columnsMask = TS.mask() | DSUMMARY.mask();

        AggrPoint[] expected = {
                point("2020-05-30T11:12:00Z", new ImmutableSummaryDoubleSnapshot(10, 1001, 1, 101, 3)),
                point("2020-05-30T11:12:15Z", new ImmutableSummaryDoubleSnapshot(20, 2001, 2, 201, 4)),
                point("2020-05-30T11:12:30Z", new ImmutableSummaryDoubleSnapshot(30, 3001, 3, 301, 5)),
                point("2020-05-30T11:12:45Z", new ImmutableSummaryDoubleSnapshot(40, 4001, 4, 401, 6)),
                point("2020-05-30T11:13:00Z", new ImmutableSummaryDoubleSnapshot(50, 5001, 5, 501, 7)),
        };

        HeapBitBuf buf = TsStreamNative.summaryDoubleEncode(columnsMask, expected);
        AggrPoint[] decoded = TsStreamJava.decode(MetricType.DSUMMARY, columnsMask, buf);
        Assert.assertArrayEquals(expected, decoded);
    }

    @Test
    public void simplePointsWithStepMillis() {
        final int columnsMask = TS.mask() | DSUMMARY.mask() | STEP.mask();

        AggrPoint[] expected = {
                pointWithStepMillis("2020-05-30T11:12:00Z", new ImmutableSummaryDoubleSnapshot(10, 1001, 1, 101, 3), 15_000),
                pointWithStepMillis("2020-05-30T11:12:15Z", new ImmutableSummaryDoubleSnapshot(20, 2001, 2, 201, 4), 30_000),
                pointWithStepMillis("2020-05-30T11:12:30Z", new ImmutableSummaryDoubleSnapshot(30, 3001, 3, 301, 5), 10_000),
                pointWithStepMillis("2020-05-30T11:12:45Z", new ImmutableSummaryDoubleSnapshot(40, 4001, 4, 401, 6), 15_000),
                pointWithStepMillis("2020-05-30T11:13:00Z", new ImmutableSummaryDoubleSnapshot(50, 5001, 5, 501, 7), 20_000),
        };

        HeapBitBuf buf = TsStreamNative.summaryDoubleEncode(columnsMask, expected);
        AggrPoint[] decoded = TsStreamJava.decode(MetricType.DSUMMARY, columnsMask, buf);
        Assert.assertArrayEquals(expected, decoded);
    }

    @Test
    public void aggrPoints() {
        final int columnsMask = TS.mask() | DSUMMARY.mask() | MERGE.mask() | COUNT.mask();

        AggrPoint[] expected = {
                point("2020-05-30T11:12:00Z", new ImmutableSummaryDoubleSnapshot(10, 1001, 1, 101, 3),  true,  1),
                point("2020-05-30T11:12:15Z", new ImmutableSummaryDoubleSnapshot(20, 2001, 2, 201, 4), false,  5),
                point("2020-05-30T11:12:30Z", new ImmutableSummaryDoubleSnapshot(30, 3001, 3, 301, 5), false,  3),
                point("2020-05-30T11:12:45Z", new ImmutableSummaryDoubleSnapshot(40, 4001, 4, 401, 6),  true,  7),
                point("2020-05-30T11:13:00Z", new ImmutableSummaryDoubleSnapshot(50, 5001, 5, 501, 7),  true, 10),
        };

        HeapBitBuf buf = TsStreamNative.summaryDoubleEncode(columnsMask, expected);
        AggrPoint[] decoded = TsStreamJava.decode(MetricType.DSUMMARY, columnsMask, buf);
        Assert.assertArrayEquals(expected, decoded);
    }

    @Test
    public void stress() {
        final int columnsMask = TS.mask() | DSUMMARY.mask() | STEP.mask() | MERGE.mask() | COUNT.mask();

        AggrPoint[] expected = new AggrPoint[1000];
        for (int i = 0; i < expected.length; i++) {
            expected[i] = AggrPointDataTestSupport.randomPoint(columnsMask);
        }

        HeapBitBuf buf = TsStreamNative.summaryDoubleEncode(columnsMask, expected);
        AggrPoint[] decoded = TsStreamJava.decode(MetricType.DSUMMARY, columnsMask, buf);
        Assert.assertArrayEquals(expected, decoded);
    }
}
