package ru.yandex.solomon.ts_codec;

import org.junit.Assert;
import org.junit.Test;

import ru.yandex.monlib.metrics.summary.ImmutableSummaryInt64Snapshot;
import ru.yandex.solomon.codec.bits.HeapBitBuf;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.AggrPointDataTestSupport;
import ru.yandex.solomon.model.protobuf.MetricType;

import static ru.yandex.solomon.model.point.AggrPoints.point;
import static ru.yandex.solomon.model.point.AggrPoints.pointWithStepMillis;
import static ru.yandex.solomon.model.point.column.StockpileColumn.COUNT;
import static ru.yandex.solomon.model.point.column.StockpileColumn.ISUMMARY;
import static ru.yandex.solomon.model.point.column.StockpileColumn.MERGE;
import static ru.yandex.solomon.model.point.column.StockpileColumn.STEP;
import static ru.yandex.solomon.model.point.column.StockpileColumn.TS;

/**
 * @author Sergey Polovko
 */
public class SummaryInt64TsDecoderTest {

    @Test
    public void simplePoints() {
        final int columnsMask = TS.mask() | ISUMMARY.mask();

        AggrPoint[] expected = {
                point("2020-05-30T11:12:00Z", new ImmutableSummaryInt64Snapshot(10, 1001, 1, 101, 3)),
                point("2020-05-30T11:12:15Z", new ImmutableSummaryInt64Snapshot(20, 2001, 2, 201, 4)),
                point("2020-05-30T11:12:30Z", new ImmutableSummaryInt64Snapshot(30, 3001, 3, 301, 5)),
                point("2020-05-30T11:12:45Z", new ImmutableSummaryInt64Snapshot(40, 4001, 4, 401, 6)),
                point("2020-05-30T11:13:00Z", new ImmutableSummaryInt64Snapshot(50, 5001, 5, 501, 7)),
        };

        HeapBitBuf buf = TsStreamJava.encode(MetricType.ISUMMARY, columnsMask, expected);
        AggrPoint[] decoded = TsStreamNative.summaryInt64Decode(columnsMask, buf);
        Assert.assertArrayEquals(expected, decoded);
    }

    @Test
    public void simplePointsWithStepMillis() {
        final int columnsMask = TS.mask() | ISUMMARY.mask() | STEP.mask();

        AggrPoint[] expected = {
                pointWithStepMillis("2020-05-30T11:12:00Z", new ImmutableSummaryInt64Snapshot(10, 1001, 1, 101, 3), 15_000),
                pointWithStepMillis("2020-05-30T11:12:15Z", new ImmutableSummaryInt64Snapshot(20, 2001, 2, 201, 4), 30_000),
                pointWithStepMillis("2020-05-30T11:12:30Z", new ImmutableSummaryInt64Snapshot(30, 3001, 3, 301, 5), 10_000),
                pointWithStepMillis("2020-05-30T11:12:45Z", new ImmutableSummaryInt64Snapshot(40, 4001, 4, 401, 6), 15_000),
                pointWithStepMillis("2020-05-30T11:13:00Z", new ImmutableSummaryInt64Snapshot(50, 5001, 5, 501, 7), 20_000),
        };

        HeapBitBuf buf = TsStreamJava.encode(MetricType.ISUMMARY, columnsMask, expected);
        AggrPoint[] decoded = TsStreamNative.summaryInt64Decode(columnsMask, buf);
        Assert.assertArrayEquals(expected, decoded);
    }

    @Test
    public void aggrPoints() {
        final int columnsMask = TS.mask() | ISUMMARY.mask() | MERGE.mask() | COUNT.mask();

        AggrPoint[] expected = {
                point("2020-05-30T11:12:00Z", new ImmutableSummaryInt64Snapshot(10, 1001, 1, 101, 3),  true,  1),
                point("2020-05-30T11:12:15Z", new ImmutableSummaryInt64Snapshot(20, 2001, 2, 201, 4), false,  5),
                point("2020-05-30T11:12:30Z", new ImmutableSummaryInt64Snapshot(30, 3001, 3, 301, 5), false,  3),
                point("2020-05-30T11:12:45Z", new ImmutableSummaryInt64Snapshot(40, 4001, 4, 401, 6),  true,  7),
                point("2020-05-30T11:13:00Z", new ImmutableSummaryInt64Snapshot(50, 5001, 5, 501, 7),  true, 10),
        };

        HeapBitBuf buf = TsStreamJava.encode(MetricType.ISUMMARY, columnsMask, expected);
        AggrPoint[] decoded = TsStreamNative.summaryInt64Decode(columnsMask, buf);
        Assert.assertArrayEquals(expected, decoded);
    }

    @Test
    public void stress() {
        final int columnsMask = TS.mask() | ISUMMARY.mask() | STEP.mask() | MERGE.mask() | COUNT.mask();

        AggrPoint[] expected = new AggrPoint[1000];
        for (int i = 0; i < expected.length; i++) {
            expected[i] = AggrPointDataTestSupport.randomPoint(columnsMask);
        }

        HeapBitBuf buf = TsStreamJava.encode(MetricType.ISUMMARY, columnsMask, expected);
        AggrPoint[] decoded = TsStreamNative.summaryInt64Decode(columnsMask, buf);
        Assert.assertArrayEquals(expected, decoded);
    }
}
