#include "combine.h"

#include <solomon/libs/cpp/ts_math/aggregation.h>
#include <solomon/libs/cpp/ts_math/error.h>

#include <solomon/libs/cpp/ts_model/visit.h>
#include <solomon/libs/cpp/proto_convert/metric_type.h>

namespace NSolomon::NTsMath {

namespace {

/*
 *  * <pre>
 *    F/T   |  DGAUGE   | IGAUGE  | COUNTER |   RATE      |   LOG_HIST  |     HIST    |
 *  -----------------------------------------------------------------------------------
 *  DGAUGE  |   same    |  round  |  round  |  integrate  |     none    |     none    |
 *  COUNTER |   cast    |  same   |  same   |  same       |     none    |     none    |
 *  RATE    |   deriv   |  same   |  same   |  same       |     none    |     none    |
 *  IGAUGE  |   cast    |  same   |  same   |  same       |     none    |     none    |
 * </pre>
 */

class TCast: public IOperation {
    template <typename TInput_, typename TOutput_>
    class TSameIterator: public NTsModel::IIterator<TOutput_> {
    public:
        using TOutput = TOutput_;
        using TInput = TInput_;

        static_assert(std::is_same_v<typename TOutput::TDataType, typename TInput::TDataType>);

    public:
        TSameIterator(THolder<NTsModel::IIterator<TInput>> underlying)
            : Underlying_{std::move(underlying)}
        {
        }

    public:
        bool NextPoint(TOutput* point) override {
            auto pointCast = std::launder(
                static_cast<TInput*>(static_cast<typename TInput::TDataType*>(point)));
            Y_ASSERT(reinterpret_cast<intptr_t>(point) == reinterpret_cast<intptr_t>(pointCast));
            return Underlying_->NextPoint(pointCast);
        }

    private:
        THolder<NTsModel::IIterator<TInput>> Underlying_;
    };

    template <typename TInput_>
    class TCastIterator: public NTsModel::IIterator<NTsModel::TGaugePoint> {
    public:
        using TOutput = NTsModel::TGaugePoint;
        using TInput = TInput_;

        static_assert(NTsModel::TPointTraits<TInput>::IsScalar);

    public:
        TCastIterator(THolder<NTsModel::IIterator<TInput>> underlying)
            : Underlying_{std::move(underlying)}
        {
        }

    public:
        bool NextPoint(TOutput* point) override {
            TInput pointCast;
            if (!Underlying_->NextPoint(&pointCast)) {
                return false;
            }
            point->Num = pointCast.Value;
            point->Denom = 0;
            static_cast<NTs::TPointCommon&>(*point) = pointCast;
            return true;
        }

    private:
        THolder<NTsModel::IIterator<TInput>> Underlying_;
    };

    template <typename TOutput_>
    class TRoundIterator: public NTsModel::IIterator<TOutput_> {
    public:
        using TOutput = TOutput_;
        using TInput = NTsModel::TGaugePoint;

        static_assert(NTsModel::TPointTraits<TOutput>::IsScalar);

    public:
        TRoundIterator(THolder<NTsModel::IIterator<TInput>> underlying)
            : Underlying_{std::move(underlying)}
        {
        }

    public:
        bool NextPoint(TOutput* point) override {
            TInput pointCast;
            if (!Underlying_->NextPoint(&pointCast)) {
                return false;
            }
            point->Value = std::round(pointCast.ValueDivided());
            static_cast<NTs::TPointCommon&>(*point) = pointCast;
            return true;
        }

    private:
        THolder<NTsModel::IIterator<TInput>> Underlying_;
    };

    class TDerivIterator: public NTsModel::IIterator<NTsModel::TGaugePoint> {
    public:
        using TOutput = NTsModel::TGaugePoint;
        using TInput = NTsModel::TRatePoint;

    public:
        TDerivIterator(THolder<NTsModel::IIterator<TInput>> underlying)
            : Underlying_{std::move(underlying)}
        {
        }

    public:
        bool NextPoint(TOutput* point) override {
            TInput cur;

            while (Underlying_->NextPoint(&cur)) {
                if (Prev_.Time == TInstant::Zero()) {
                    Prev_ = cur;
                    continue;
                }

                if (!IsResetRequired(cur)) {
                    DeltaValue_ = cur.Value - Prev_.Value;
                    DeltaTime_ = (cur.Time - Prev_.Time).MilliSeconds();
                    LastCount_ = cur.Count;
                }

                Prev_ = cur;

                static_cast<NTs::TPointCommon&>(*point) = cur;
                point->Num = DeltaValue_;
                point->Denom = DeltaTime_;
                point->Count = LastCount_;

                return true;
            }

            return false;
        }

    private:
        bool IsResetRequired(TInput cur) {
            if (cur.Value < Prev_.Value) {
                return true;
            }

            if (cur.Count != Prev_.Count) {
                return true;
            }

            if (cur.Time < Prev_.Time || cur.Time > Prev_.Time + TDuration::Days(7)) {
                return true;
            }

            return Prev_.Step != TDuration::Zero() && cur.Time - Prev_.Time > Prev_.Step;
        }

    private:
        THolder<NTsModel::IIterator<TInput>> Underlying_;
        TInput Prev_;
        ui64 LastCount_ = 0;
        double DeltaValue_ = 0;
        ui64 DeltaTime_ = 0;
    };

    class TIntegrateIterator: public NTsModel::IIterator<NTsModel::TRatePoint> {
    public:
        using TOutput = NTsModel::TRatePoint;
        using TInput = NTsModel::TGaugePoint;
    };

    template <typename TIterator>
    class TIterable: public NTsModel::IIterable {
    public:
        TIterable(THolder<NTsModel::IIterable> underlying)
            : Underlying_{std::move(underlying)}
        {
        }

    public:
        NTsModel::EPointType Type() const override {
            return TIterator::TOutput::Type;
        }

        NTsModel::TPointColumns Columns() const override {
            return Underlying_->Columns();
        }

        THolder<NTsModel::IGenericIterator> Iterator() const override {
            return THolder<NTsModel::IGenericIterator>{Underlying_->Iterator().Release()->As<typename TIterator::TInput>()};
        }

        TInstant WindowBegin() const override {
            return Underlying_->WindowBegin();
        }

        TInstant WindowEnd() const override {
            return Underlying_->WindowEnd();
        }

    private:
        THolder<NTsModel::IIterable> Underlying_;
    };

    struct TConvertFrom {
        THolder<NTsModel::IIterable> operator()(
                NTsModel::TPointTraits<NTsModel::TGaugePoint>, THolder<NTsModel::IIterable> src) const
        {
            switch (src->Type()) {
                case NTsModel::EPointType::Unknown:
                    return nullptr;
                case NTsModel::EPointType::DGauge:
                    return MakeHolder<TIterable<TSameIterator<NTsModel::TGaugePoint, NTsModel::TGaugePoint>>>(std::move(src));
                case NTsModel::EPointType::Counter:
                    return MakeHolder<TIterable<TCastIterator<NTsModel::TCounterPoint>>>(std::move(src));
                case NTsModel::EPointType::Rate:
                    return MakeHolder<TIterable<TDerivIterator>>(std::move(src));
                case NTsModel::EPointType::IGauge:
                    return MakeHolder<TIterable<TCastIterator<NTsModel::TIGaugePoint>>>(std::move(src));
                case NTsModel::EPointType::Hist:
                    return nullptr;
                case NTsModel::EPointType::HistRate:
                    return nullptr;
                case NTsModel::EPointType::DSummary:
                    return nullptr;
                case NTsModel::EPointType::LogHist:
                    return nullptr;
            }
        }

        THolder<NTsModel::IIterable> operator()(
                NTsModel::TPointTraits<NTsModel::TCounterPoint>, THolder<NTsModel::IIterable> src) const
        {
            switch (src->Type()) {
                case NTsModel::EPointType::Unknown:
                    return nullptr;
                case NTsModel::EPointType::DGauge:
                    return MakeHolder<TIterable<TRoundIterator<NTsModel::TCounterPoint>>>(std::move(src));
                case NTsModel::EPointType::Counter:
                    return MakeHolder<TIterable<TSameIterator<NTsModel::TCounterPoint, NTsModel::TCounterPoint>>>(std::move(src));
                case NTsModel::EPointType::Rate:
                    return MakeHolder<TIterable<TSameIterator<NTsModel::TRatePoint, NTsModel::TCounterPoint>>>(std::move(src));
                case NTsModel::EPointType::IGauge:
                    return MakeHolder<TIterable<TSameIterator<NTsModel::TIGaugePoint, NTsModel::TCounterPoint>>>(std::move(src));
                case NTsModel::EPointType::Hist:
                    return nullptr;
                case NTsModel::EPointType::HistRate:
                    return nullptr;
                case NTsModel::EPointType::DSummary:
                    return nullptr;
                case NTsModel::EPointType::LogHist:
                    return nullptr;
            }
        }

        THolder<NTsModel::IIterable> operator()(
                NTsModel::TPointTraits<NTsModel::TRatePoint>, THolder<NTsModel::IIterable> src) const
        {
            switch (src->Type()) {
                case NTsModel::EPointType::Unknown:
                    return nullptr;
                case NTsModel::EPointType::DGauge:
                    return nullptr; // TODO: integrate
                case NTsModel::EPointType::Counter:
                    return MakeHolder<TIterable<TSameIterator<NTsModel::TCounterPoint, NTsModel::TRatePoint>>>(std::move(src));
                case NTsModel::EPointType::Rate:
                    return MakeHolder<TIterable<TSameIterator<NTsModel::TRatePoint, NTsModel::TRatePoint>>>(std::move(src));
                case NTsModel::EPointType::IGauge:
                    return MakeHolder<TIterable<TSameIterator<NTsModel::TIGaugePoint, NTsModel::TRatePoint>>>(std::move(src));
                case NTsModel::EPointType::Hist:
                    return nullptr;
                case NTsModel::EPointType::HistRate:
                    return nullptr;
                case NTsModel::EPointType::DSummary:
                    return nullptr;
                case NTsModel::EPointType::LogHist:
                    return nullptr;
            }
        }

        THolder<NTsModel::IIterable> operator()(
                NTsModel::TPointTraits<NTsModel::TIGaugePoint>, THolder<NTsModel::IIterable> src) const
        {
            switch (src->Type()) {
                case NTsModel::EPointType::Unknown:
                    return nullptr;
                case NTsModel::EPointType::DGauge:
                    return MakeHolder<TIterable<TRoundIterator<NTsModel::TIGaugePoint>>>(std::move(src));
                case NTsModel::EPointType::Counter:
                    return MakeHolder<TIterable<TSameIterator<NTsModel::TCounterPoint, NTsModel::TIGaugePoint>>>(std::move(src));
                case NTsModel::EPointType::Rate:
                    return MakeHolder<TIterable<TSameIterator<NTsModel::TRatePoint, NTsModel::TIGaugePoint>>>(std::move(src));
                case NTsModel::EPointType::IGauge:
                    return MakeHolder<TIterable<TSameIterator<NTsModel::TIGaugePoint, NTsModel::TIGaugePoint>>>(std::move(src));
                case NTsModel::EPointType::Hist:
                    return nullptr;
                case NTsModel::EPointType::HistRate:
                    return nullptr;
                case NTsModel::EPointType::DSummary:
                    return nullptr;
                case NTsModel::EPointType::LogHist:
                    return nullptr;
            }
        }

        THolder<NTsModel::IIterable> operator()(
                NTsModel::TPointTraits<NTsModel::THistPoint>, THolder<NTsModel::IIterable> src) const
        {
            switch (src->Type()) {
                case NTsModel::EPointType::Unknown:
                    return nullptr;
                case NTsModel::EPointType::DGauge:
                    return nullptr;
                case NTsModel::EPointType::Counter:
                    return nullptr;
                case NTsModel::EPointType::Rate:
                    return nullptr;
                case NTsModel::EPointType::IGauge:
                    return nullptr;
                case NTsModel::EPointType::Hist:
                    return MakeHolder<TIterable<TSameIterator<NTsModel::THistPoint, NTsModel::THistPoint>>>(std::move(src));
                case NTsModel::EPointType::HistRate:
                    return MakeHolder<TIterable<TSameIterator<NTsModel::THistPoint, NTsModel::THistRatePoint>>>(std::move(src));
                case NTsModel::EPointType::DSummary:
                    return nullptr;
                case NTsModel::EPointType::LogHist:
                    return nullptr; // TODO: convert
            }
        }

        THolder<NTsModel::IIterable> operator()(
                NTsModel::TPointTraits<NTsModel::THistRatePoint>, THolder<NTsModel::IIterable> src) const
        {
            switch (src->Type()) {
                case NTsModel::EPointType::Unknown:
                    return nullptr;
                case NTsModel::EPointType::DGauge:
                    return nullptr;
                case NTsModel::EPointType::Counter:
                    return nullptr;
                case NTsModel::EPointType::Rate:
                    return nullptr;
                case NTsModel::EPointType::IGauge:
                    return nullptr;
                case NTsModel::EPointType::Hist:
                    return MakeHolder<TIterable<TSameIterator<NTsModel::THistRatePoint, NTsModel::THistPoint>>>(std::move(src));
                case NTsModel::EPointType::HistRate:
                    return MakeHolder<TIterable<TSameIterator<NTsModel::THistRatePoint, NTsModel::THistRatePoint>>>(std::move(src));
                case NTsModel::EPointType::DSummary:
                    return nullptr;
                case NTsModel::EPointType::LogHist:
                    return nullptr; // TODO: convert
            }
        }

        THolder<NTsModel::IIterable> operator()(
                NTsModel::TPointTraits<NTsModel::TDSummaryPoint>, THolder<NTsModel::IIterable> src) const
        {
            switch (src->Type()) {
                case NTsModel::EPointType::Unknown:
                    return nullptr;
                case NTsModel::EPointType::DGauge:
                    return nullptr;
                case NTsModel::EPointType::Counter:
                    return nullptr;
                case NTsModel::EPointType::Rate:
                    return nullptr;
                case NTsModel::EPointType::IGauge:
                    return nullptr;
                case NTsModel::EPointType::Hist:
                    return nullptr;
                case NTsModel::EPointType::HistRate:
                    return nullptr;
                case NTsModel::EPointType::DSummary:
                    return MakeHolder<TIterable<TSameIterator<NTsModel::TDSummaryPoint, NTsModel::TDSummaryPoint>>>(std::move(src));
                case NTsModel::EPointType::LogHist:
                    return nullptr;
            }
        }

        THolder<NTsModel::IIterable> operator()(
                NTsModel::TPointTraits<NTsModel::TLogHistPoint>, THolder<NTsModel::IIterable> src) const
        {
            switch (src->Type()) {
                case NTsModel::EPointType::Unknown:
                    return nullptr;
                case NTsModel::EPointType::DGauge:
                    return nullptr;
                case NTsModel::EPointType::Counter:
                    return nullptr;
                case NTsModel::EPointType::Rate:
                    return nullptr;
                case NTsModel::EPointType::IGauge:
                    return nullptr;
                case NTsModel::EPointType::Hist:
                    return nullptr; // TODO: convert
                case NTsModel::EPointType::HistRate:
                    return nullptr; // TODO: convert
                case NTsModel::EPointType::DSummary:
                    return nullptr;
                case NTsModel::EPointType::LogHist:
                    return MakeHolder<TIterable<TSameIterator<NTsModel::TLogHistPoint, NTsModel::TLogHistPoint>>>(std::move(src));
            }
        }
    };

public:
    TCast(NTsModel::EPointType type)
        : Type_{type}
    {
    }

public:
    TVector<TTimeSeries> Apply(TVector<TTimeSeries>&& source) override {
        for (auto& ts: source) {
            if (ts.Data->Type() == Type_) {
                continue;
            }

            auto type = ts.Data->Type();
            auto iterable = NTsModel::Visit(Type_, TConvertFrom{}, std::move(ts.Data));
            if (!iterable) {
                ythrow TTypeError{} << "unable to cast from " << type << " to " << Type_;
            }
            ts.Data = std::move(iterable);
        }

        return std::move(source);
    }

private:
    NTsModel::EPointType Type_;
};

} // namespace

THolder<IOperation> Cast(const yandex::solomon::math::OperationCast& settings) {
    auto type = NTsModel::FromProto(settings.type());
    if (type.Fail()) {
        ythrow TConfigError{} << type.Error();
    }
    return MakeHolder<TCast>(type.Value());
}

} // namespace NSolomon::NTsMath
