#include "downsampling.h"

#include <solomon/libs/cpp/ts_math/aggregation.h>
#include <solomon/libs/cpp/ts_model/aggregator.h>
#include <solomon/libs/cpp/ts_model/aggregator_avg.h>
#include <solomon/libs/cpp/ts_model/aggregator_count.h>
#include <solomon/libs/cpp/ts_model/aggregator_last.h>
#include <solomon/libs/cpp/ts_model/aggregator_min.h>
#include <solomon/libs/cpp/ts_model/aggregator_max.h>
#include <solomon/libs/cpp/ts_model/aggregator_sum.h>
#include <solomon/libs/cpp/ts_math/error.h>
#include <solomon/libs/cpp/ts_model/visit.h>

#include <util/generic/queue.h>

namespace NSolomon::NTsMath {

namespace {

using EFillOption = yandex::solomon::math::OperationDownsampling_FillOption;

class TDownsampling: public IOperation {
    template <typename TAggregator>
    class TIterator: public NTsModel::IIterator<typename TAggregator::TOutput> {
        using TInput = typename TAggregator::TInput;
        using TOutput = typename TAggregator::TOutput;

    public:
        TIterator(
                THolder<NTsModel::IIterator<TInput>> underlying,
                yandex::solomon::math::OperationDownsampling settings,
                TInstant windowBegin,
                TInstant windowEnd)
            : Underlying_{std::move(underlying)}
            , Settings_{std::move(settings)}
            , WindowBegin_{windowBegin}
            , WindowEnd_{windowEnd}
        {
            Y_VERIFY(Settings_.grid_millis() != 0, "Cannot use downsampling with grid_millis = 0");
            HasCur_ = Underlying_->NextPoint(&Cur_);
        }

    public:
        bool NextPoint(TOutput* point) override {
            while (true) {
                if (WindowBegin_ > WindowEnd_) {
                    return false;
                }

                auto windowSize = TDuration::MilliSeconds(Settings_.grid_millis());
                auto curWindowBegin = WindowBegin_;
                auto curWindowEnd = curWindowBegin + windowSize;

                WindowBegin_ += windowSize;

                while (HasCur_ && Cur_.Time < curWindowEnd) {
                    LastAddedPointTs_ = Cur_.Time;
                    LastAddedPointStep_ = Cur_.Step;
                    Aggregator_.Add(Cur_);
                    HasCur_ = Underlying_->NextPoint(&Cur_);
                }

                if (Aggregator_.HasPoints()) {
                    Prev_ = Aggregator_.Finish();
                    Prev_.Time = curWindowBegin;

                    *point = Prev_;
                    return true;
                } else {
                    Prev_.Time = curWindowBegin;
                    Prev_.Count = 0;
                }

                // If downsampling interval is lesser than step, we'll have gaps between points which we don't want
                // to report. For example, if downsampling interval is five seconds, but points are reported once in
                // ten seconds, every other point produced by downsampling will be none. This behaviour is unwanted
                // because it will interfere with graph rendering. See SOLOMON-3575 for details.
                if (
                    !Settings_.ignore_min_step_millis() &&
                    LastAddedPointTs_ > TInstant::Zero() &&
                    curWindowEnd <= LastAddedPointTs_ + LastAddedPointStep_)
                {
                    continue; // move to next window
                }

                switch (Settings_.fill_option()) {
                    default:
                    case yandex::solomon::math::OperationDownsampling::NULL_:
                        // XXX: for now, only gauge points can have nan values.
                        //      Other points are not reported.
                        if constexpr (std::is_same_v<TOutput, NTsModel::TGaugePoint>) {
                            *point = {};
                            point->Num = std::numeric_limits<double>::quiet_NaN();
                            point->Denom = 0;
                            point->Time = curWindowBegin;
                            return true;
                        }
                        break; // move to next window
                    case yandex::solomon::math::OperationDownsampling::NONE:
                        break; // move to next window
                    case yandex::solomon::math::OperationDownsampling::PREVIOUS:
                        *point = Prev_;
                        return true;
                }
            }
        }

    private:
        THolder<NTsModel::IIterator<TInput>> Underlying_;
        yandex::solomon::math::OperationDownsampling Settings_;
        TInstant WindowBegin_;
        TInstant WindowEnd_;
        TAggregator Aggregator_ = {};

        TInstant LastAddedPointTs_;
        TDuration LastAddedPointStep_;

        TInput Cur_;
        bool HasCur_ = false;

        TOutput Prev_;
    };

    class TIterable: public NTsModel::IIterable {
    public:
        TIterable(THolder<NTsModel::IIterable> underlying, yandex::solomon::math::OperationDownsampling settings)
            : Underlying_{std::move(underlying)}
            , Settings_{std::move(settings)}
            , WindowBegin_{Underlying_->WindowBegin()}
            , WindowEnd_{Underlying_->WindowEnd()}
        {
            Y_VERIFY(Settings_.grid_millis() != 0, "Cannot use downsampling with grid_millis = 0");
            WindowBegin_ -= TDuration::MilliSeconds(WindowBegin_.MilliSeconds() % Settings_.grid_millis());
            WindowEnd_ -= TDuration::MilliSeconds(WindowEnd_.MilliSeconds() % Settings_.grid_millis());

            auto numPoints = (WindowEnd_ - WindowBegin_).MilliSeconds() / Settings_.grid_millis();
            if (numPoints > TDuration::Days(360).Minutes() / 5) {
                ythrow TConstraintError{}
                    << "requested downsampling step " << TDuration::MilliSeconds(Settings_.grid_millis()) << " "
                    << "is too small for time window [" << WindowBegin_ << " .. " << WindowEnd_ << "] "
                    << "and will produce " << numPoints << " points";
            }
        }

        NTsModel::EPointType Type() const override {
            return Underlying_->Type();
        }

        NTsModel::TPointColumns Columns() const override {
            // TODO: these columns should come from IAggregator::Columns
            return NTsModel::TPointColumns{NTsModel::TPointColumns::Step, NTsModel::TPointColumns::Count};
        }

        THolder<NTsModel::IGenericIterator> Iterator() const override {
            auto aggregation = AggregationFunction(Settings_.aggregation(), Underlying_->Type());

            return NTsModel::Visit(Underlying_->Type(), [this, aggregation](auto traits) -> THolder<NTsModel::IGenericIterator> {
                using TPoint = typename decltype(traits)::TPoint;
                auto iterator = traits.DowncastIterator(Underlying_->Iterator());

                if (aggregation == NTsModel::EAggregationFunction::Count) {
                    return MakeHolder<TIterator<NTsModel::TCountAggregator<TPoint>>>(
                        std::move(iterator), Settings_, WindowBegin_, WindowEnd_);
                }

                if (AggregateNonScalar(aggregation)) {
                    switch (aggregation) {
                        case NTsModel::EAggregationFunction::Last:
                            return MakeHolder<TIterator<NTsModel::TLastAggregator<TPoint>>>(
                                std::move(iterator), Settings_, WindowBegin_, WindowEnd_);
                        case NTsModel::EAggregationFunction::Sum:
                            return MakeHolder<TIterator<NTsModel::TSumAggregator<TPoint>>>(
                                std::move(iterator), Settings_, WindowBegin_, WindowEnd_);
                        case NTsModel::EAggregationFunction::Count:
                            return MakeHolder<TIterator<NTsModel::TCountAggregator<TPoint>>>(
                                std::move(iterator), Settings_, WindowBegin_, WindowEnd_);
                        default:
                            ythrow TTypeError{} << "unknown aggregation function: " << aggregation;
                    }
                } else {
                    if constexpr (traits.IsScalar) {
                        switch (aggregation) {
                            case NTsModel::EAggregationFunction::Min:
                                return MakeHolder<TIterator<NTsModel::TMinAggregator<TPoint>>>(
                                    std::move(iterator), Settings_, WindowBegin_, WindowEnd_);
                            case NTsModel::EAggregationFunction::Max:
                                return MakeHolder<TIterator<NTsModel::TMaxAggregator<TPoint>>>(
                                    std::move(iterator), Settings_, WindowBegin_, WindowEnd_);
                            case NTsModel::EAggregationFunction::Avg:
                                return MakeHolder<TIterator<NTsModel::TAvgAggregator<TPoint>>>(
                                    std::move(iterator), Settings_, WindowBegin_, WindowEnd_);
                            default:
                                ythrow TTypeError{} << "unknown aggregation function: " << aggregation;
                        }
                    } else {
                        ythrow TTypeError{}
                            << "unable to perform downsampling with function " << aggregation << " "
                            << "for time series of type " << traits.Type();
                    }
                }
            });
        }

        TInstant WindowBegin() const override {
            return WindowBegin_;
        }

        TInstant WindowEnd() const override {
            return WindowEnd_;
        }

    private:
        THolder<NTsModel::IIterable> Underlying_;
        yandex::solomon::math::OperationDownsampling Settings_;
        TInstant WindowBegin_;
        TInstant WindowEnd_;
    };

public:
    TDownsampling(yandex::solomon::math::OperationDownsampling settings)
        : Settings_{std::move(settings)}
    {
    }

public:
    TVector<TTimeSeries> Apply(TVector<TTimeSeries>&& source) override {
        // Do not apply if grid_middis doesn't set
        if (Settings_.grid_millis() != 0) {
            for (auto& ts: source) {
                ts.Data.Reset(new TIterable{std::move(ts.Data), Settings_});
            }
        }
        return std::move(source);
    }

private:
    yandex::solomon::math::OperationDownsampling Settings_;
};

} // namespace

THolder<IOperation> Downsampling(const yandex::solomon::math::OperationDownsampling& settings) {
    return MakeHolder<TDownsampling>(settings);
}

} // namespace NSolomon::NTsMath
