#include "top.h"

#include <solomon/libs/cpp/ts_math/aggregation.h>
#include <solomon/libs/cpp/ts_model/aggregator.h>
#include <solomon/libs/cpp/ts_model/aggregator_avg.h>
#include <solomon/libs/cpp/ts_model/aggregator_count.h>
#include <solomon/libs/cpp/ts_model/aggregator_last.h>
#include <solomon/libs/cpp/ts_model/aggregator_min.h>
#include <solomon/libs/cpp/ts_model/aggregator_max.h>
#include <solomon/libs/cpp/ts_model/aggregator_sum.h>
#include <solomon/libs/cpp/ts_math/error.h>

#include <solomon/libs/cpp/ts_model/visit.h>
#include <solomon/libs/cpp/proto_convert/metric_type.h>

#include <util/generic/queue.h>
#include <util/generic/set.h>

namespace NSolomon::NTsMath {

namespace {

template <typename TAggregator>
double Aggregate(THolder<NTsModel::IIterator<typename TAggregator::TInput>> iterator) {
    TAggregator aggregator;
    typename TAggregator::TInput point;
    while (iterator->NextPoint(&point)) {
        aggregator.Add(point);
    }

    if constexpr (std::is_same_v<typename TAggregator::TOutput, NTsModel::TGaugePoint>) {
        return aggregator.Finish().ValueDivided();
    } else {
        return aggregator.Finish().Value;
    }
}

class TTop: public IOperation {
public:
    TTop(yandex::solomon::math::Aggregation aggregation, size_t limit, bool asc)
        : Aggregation_{aggregation}
        , Limit_{limit}
        , Asc_{asc}
    {
    }

public:
    TVector<TTimeSeries> Apply(TVector<TTimeSeries>&& source) override {
        TMultiSet<std::pair<double, TTimeSeries*>> top;

        for (auto& ts: source) {
            auto aggregation = AggregationFunction(Aggregation_, ts.Data->Type());
            auto summary = NTsModel::Visit(ts.Data->Type(), [aggregation, &ts](auto traits) -> double {
                using TPoint = typename decltype(traits)::TPoint;
                auto iterator = traits.DowncastIterator(ts.Data->Iterator());

                if (aggregation == NTsModel::EAggregationFunction::Count) {
                    return Aggregate<NTsModel::TCountAggregator<TPoint>>(std::move(iterator));
                }

                if constexpr (!traits.IsScalar) {
                    ythrow TTypeError{} << "unable to build top for time series of type " << traits.Type();
                } else {
                    switch (aggregation) {
                        case NTsModel::EAggregationFunction::Min:
                            return Aggregate<NTsModel::TMinAggregator<TPoint>>(std::move(iterator));
                        case NTsModel::EAggregationFunction::Max:
                            return Aggregate<NTsModel::TMaxAggregator<TPoint>>(std::move(iterator));
                        case NTsModel::EAggregationFunction::Sum:
                            return Aggregate<NTsModel::TSumAggregator<TPoint>>(std::move(iterator));
                        case NTsModel::EAggregationFunction::Last:
                            return Aggregate<NTsModel::TLastAggregator<TPoint>>(std::move(iterator));
                        case NTsModel::EAggregationFunction::Avg:
                            return Aggregate<NTsModel::TAvgAggregator<TPoint>>(std::move(iterator));
                        case NTsModel::EAggregationFunction::Count:
                            return Aggregate<NTsModel::TCountAggregator<TPoint>>(std::move(iterator));
                    }
                }
            });

            if (!Asc_) {
                summary = -summary;
            }

            top.insert({summary, &ts});
            if (top.size() > Limit_) {
                top.erase(top.begin());
            }
        }

        TVector<TTimeSeries> res;
        res.reserve(top.size());
        for (auto& [_, ts]: top) {
            res.push_back(std::move(*ts));
        }
        return res;
    }

private:
    yandex::solomon::math::Aggregation Aggregation_;
    size_t Limit_;
    bool Asc_;
};

} // namespace

THolder<IOperation> Top(const yandex::solomon::math::OperationTop& settings) {
    return MakeHolder<TTop>(settings.time_aggregation(), settings.limit(), settings.asc());
}

} // namespace NSolomon::NTsMath
