#pragma once

#include "aggregation.h"

#include <solomon/libs/cpp/ts_model/iterator.h>
#include <solomon/libs/cpp/ts_model/points.h>

#include <solomon/protos/math/aggregation.pb.h>

#include <optional>
#include <variant>

namespace NSolomon::NTsMath {

/**
 * Summary of a time series.
 */
class TSummary {
public:
    static TSummary Calculate(NTsModel::IGenericIterator&& iterator);

public:
    /**
     * Get data type of this summary.
     */
    NTsModel::EPointType Type() const;

    /**
     * Compare summaries by the given aggregation function.
     * If summaries can't be compared, calling this function causes an undefined behaviour.
     */
    bool LesserThen(EAggregation aggregation, const TSummary& rhs) const;

private:
    template <typename T>
    struct TScalar {
        T Sum = {};
        T Min = {};
        T Max = {};
        T Last = {};
    };

    template <typename T>
    struct TComplex {
        std::optional<T> Sum = {};
        std::optional<T> Last = {};
    };

    using TValueDouble = TScalar<double>;
    using TValueInt = TScalar<i64>;
    using TValueHist = TComplex<NTs::NValue::THistogram>;
    using TValueLogHist = TComplex<NTs::NValue::TLogHistogram>;
    using TValueDSummary = TComplex<NTs::NValue::TSummaryDouble>;
    using TValue = std::variant<TValueDouble, TValueInt, TValueHist, TValueLogHist, TValueDSummary>;

    NTsModel::EPointType Type_;
    i64 Count_;
    TValue Data_;

    inline double Max() const;
    inline double Min() const;
    inline double Sum() const;
    inline double Avg() const;
    inline double Last() const;
    inline double Count() const;
};

/**
 * Check if we can compare summaries by the given dimension.
 * We can compare any summaries by count. We can compare scalar values by all other dimensions.
 */
bool CanCompare(EAggregation aggregation, NTsModel::EPointType lhs, NTsModel::EPointType rhs);
bool CanCompare(EAggregation aggregation, const TSummary& lhs, const TSummary& rhs);

} // namespace NSolomon::NTsMath
