#pragma once

#include "aggregator.h"
#include "merge.h"

namespace NSolomon::NTsModel {

/**
 * Aggregate points by adding their values together.
 */
template <typename TPoint>
class TSumAggregator final: public TAggregator<TPoint> {
public:
    void Add(const TPoint& point) override {
        this->AddAggr(point);
        AddValue(point);
    }

    void Add(const NTs::TVariantPoint& point) {
        this->AddAggr(point);
        AddValue(point.Get<typename TPoint::TValue>());
    }

    void AddValue(const typename TPoint::TValue& value) {
        Tmp = value; // TODO: accept `point` by rvalue ref and move?
        MergeRaw(State, &Tmp);
        std::swap(State, Tmp);
    }

    TPoint Finish() override {
        TPoint res;
        static_cast<typename TPoint::TValue&>(res) = std::move(State);
        this->FillAggr(&res);
        this->Reset();
        return res;
    }

    EAggregationFunction Function() const override {
        return EAggregationFunction::Sum;
    }

    void Reset() override {
        State = {};
        this->ResetAggr();
    }

public:
    typename TPoint::TValue State = {};
    typename TPoint::TValue Tmp = {};
};

template <>
class TSumAggregator<TGaugePoint> final: public TAggregator<TGaugePoint> {
public:
    void Add(const TGaugePoint& point) override {
        this->AddAggr(point);
        AddValue(point);
    }

    void Add(const NTs::TVariantPoint& point) {
        this->AddAggr(point);
        AddValue(point.Get<TGaugePoint::TValue>());
    }

    void AddValue(const TGaugePoint::TValue& value) {
        // use Kahan summation algorithm for doubles
        double compensated = value.ValueDivided() - Compensation;
        double sum = Sum + compensated;
        Compensation = (sum - Sum) - compensated;
        Sum = sum;
    }

    TGaugePoint Finish() override {
        TGaugePoint res;
        double num = Sum + Compensation;
        if (std::isnan(num) && std::isinf(Sum)) {
            // If the compensated sum is spuriously NaN from accumulating
            // one or more same-signed infinite values,
            // use the correctly-signed infinity stored in Sum_.
            res.Num = Sum;
        } else {
            res.Num = num;
        }
        res.Denom = 0;
        this->FillAggr(&res);
        this->Reset();
        return res;
    }

    EAggregationFunction Function() const override {
        return EAggregationFunction::Sum;
    }

    void Reset() override {
        Sum = 0;
        Compensation = 0;
        this->ResetAggr();
    }

public:
    double Sum = 0;
    double Compensation = 0;
};

} // namespace NSolomon::NTsModel
