#pragma once

#include "iterator.h"
#include "point_columns.h"

namespace NSolomon::NTsModel {

/**
 * An entity capable of yielding multiple iterators over its contents.
 */
class IIterable {
public:
    virtual ~IIterable() = default;

public:
    /**
     * Get type of iterators returned by this iterable.
     */
    virtual EPointType Type() const = 0;

    /**
     * Which columns are set in the points from this iterable.
     */
    virtual TPointColumns Columns() const = 0;

    /**
     * Create a new iterator over this entity's contents.
     */
    virtual THolder<IGenericIterator> Iterator() const = 0;

    /**
     * Beginning of the time window that constraints points in this iterable.
     *
     * All points from this iterable will have timestamp no less than the value returned from this function.
     *
     * Could return `TInstant::Zero()` for data sources without known windows.
     * Could return a value greater than `WindowEnd`, in this case the returned iterables are empty.
     */
    virtual TInstant WindowBegin() const = 0;

    /**
     * Ending of the time window that constraints points in this iterable.
     *
     * All points from this iterable will have timestamp no greater than the value returned from this function.
     *
     * Could return `TInstant::Max()` for data sources without known windows.
     * Could return a value lesser than `WindowBegin`, in this case the returned iterables are empty.
     */
    virtual TInstant WindowEnd() const = 0;
};

} // namespace NSolomon::NTsModel
