#pragma once

#include "points.h"

namespace NSolomon::NTsModel {

namespace NPrivate {

struct TIteratorTag {};

} // NPrivate

template <typename TPoint_>
class IIterator;

/**
 * Abstract base class for iterators of different types.
 */
class IGenericIterator {
public:
    IGenericIterator(IGenericIterator&&) noexcept = default;
    IGenericIterator& operator=(IGenericIterator&&) noexcept = default;

    IGenericIterator(const IGenericIterator&) = delete;
    IGenericIterator& operator=(const IGenericIterator&) = delete;

    IGenericIterator() = default;
    virtual ~IGenericIterator() = default;

public:
    /**
     * Get type of this iterator.
     */
    virtual EPointType Type() const = 0;

    /**
     * Cast this iterator base to a concrete iterator type.
     * Panic if requested type doesn't match type of this iterator.
     */
    template <typename TPoint>
    IIterator<TPoint>* As() noexcept {
        Y_VERIFY(Type() == TPoint::Type, "iterator cast failed");
        return CheckedCast<IIterator<TPoint>*>(this);
    }
    template <typename TPoint>
    const IIterator<TPoint>* As() const noexcept {
        Y_VERIFY(Type() == TPoint::Type, "iterator cast failed");
        return CheckedCast<const IIterator<TPoint>*>(this);
    }
};

/**
 * Base class for iterating over time series.
 */
template <typename TPoint_>
class IIterator: public IGenericIterator, public NPrivate::TIteratorTag {
    static_assert(IsPointV<TPoint_>, "expected a time series point");

public:
    using TPoint = TPoint_;

public:
    EPointType Type() const final {
        return TPoint::Type;
    }

public:
    /**
     * Load next point from the reader.
     *
     * If there is no point to load, returns false. When this happens, `point` may be changed,
     * but is guaranteed to be valid.
     */
    [[nodiscard]]
    virtual bool NextPoint(TPoint* point) = 0;
};

/**
 * Check if the given type `T` is an iterator.
 */
template <typename T>
static constexpr const bool IsIteratorV = std::is_base_of_v<NPrivate::TIteratorTag, T>;

/**
 * Check if the given type `T` is an iterator that yields points of type `TPoint`.
 */
template <typename T, typename TPoint>
static constexpr const bool IsIteratorPointV = std::is_base_of_v<IIterator<TPoint>, T>;

} // namespace NSolomon::NTsModel
