#pragma once

#include "iterator.h"
#include "iterator_cat.h"
#include "merge.h"

#include <util/generic/array_ref.h>
#include <util/generic/vector.h>
#include <util/generic/deque.h>

namespace NSolomon::NTsModel {

/**
 * Merge points from multiple ranges.
 */
template <typename TIt>
class TMergeIterator final: public IIterator<typename TIt::TPoint> {
    static_assert(IsIteratorV<TIt>, "expected an iterator");

public:
    struct TWindow {
        TIt Iterator;
        TInstant Begin = TInstant::Zero();
        TInstant End = TInstant::Max();
    };

public:
    TMergeIterator() = default;

    template <typename T>
    TMergeIterator(T begin, T end) {
        TInstant lastPoint = TInstant::Zero();

        Iterators_.reserve(std::distance(begin, end));

        while (begin != end) {
            auto& window = *begin++;

            if (Iterators_.empty() || lastPoint > window.Begin) {
                auto& elem = Iterators_.emplace_back(typename TIt::TPoint{}, TCatIterator<TIt>());
                if (!window.Iterator.NextPoint(&elem.first)) {
                    Iterators_.pop_back();
                } else {
                    elem.second.Append(std::move(window.Iterator));
                    lastPoint = window.End;
                }
            } else {
                Iterators_.back().second.Append(std::move(window.Iterator));
                lastPoint = window.End;
            }
        }

        FindMinPoint();
    }

public:
    bool NextPoint(typename TIt::TPoint* point) override {
        if (!HasMinPoint()) {
            return false;
        }

        PopMinPoint(point, false);

        while (true) {
            FindMinPoint();

            if (!HasMinPoint() || Iterators_[MinPointIdx_].first.Time != point->Time) {
                break;
            }

            PopMinPoint(point, true);
        }

        return true;
    }

private:
    void FindMinPoint() {
        if (!Iterators_.empty()) {
            TInstant minTime = Iterators_.front().first.Time;
            MinPointIdx_ = 0;

            for (size_t i = 1; i < Iterators_.size(); ++i) {
                if (Iterators_[i].first.Time < minTime) {
                    minTime = Iterators_[i].first.Time;
                    MinPointIdx_ = i;
                }
            }
        } else {
            MinPointIdx_ = Max<size_t>();
        }
    }

    bool HasMinPoint() const {
        return MinPointIdx_ != Max<size_t>();
    }

    void PopMinPoint(typename TIt::TPoint* point, bool merge) {
        auto& [pendingPoint, iterator] = Iterators_[MinPointIdx_];

        if (merge) {
            Merge(*point, &pendingPoint);
        }

        std::swap(pendingPoint, *point);
        if (!iterator.NextPoint(&pendingPoint)) {
            std::swap(Iterators_[MinPointIdx_], Iterators_.back());
            Iterators_.pop_back();
        }
    }

private:
    TVector<std::pair<typename TIt::TPoint, TCatIterator<TIt>>> Iterators_;
    size_t MinPointIdx_ = Max<size_t>();
};

} // namespace NSolomon::NTsModel
