#pragma once

#include "points.h"
#include "point_type.h"
#include <solomon/libs/cpp/ts_codec/columns.h>

#include <util/system/compat.h>
#include <util/system/types.h>
#include <util/system/yassert.h>

#include <bitset>

namespace NSolomon::NTsModel {

/**
 * This class contains information about which point columns are filled.
 *
 * Not all metrics have the same set of columns. For example, non-aggregate metrics do not have `Merge` and `Count`
 * columns. This helps to save some space in long term storage.
 */
class TPointColumns {
public:
    enum EColumn: ui8 {
        Step,
        Merge,
        Count,

        NumColumns,
    };

private:
    using TBitSet = std::bitset<NumColumns>;

public:
    TPointColumns() = default;
    TPointColumns(std::initializer_list<EColumn> columns) {
        for (auto column: columns) {
            SetColumn(column);
        }
    }

private:
    explicit constexpr TPointColumns(TBitSet mask)
        : Mask_{mask}
    {
    }

private:
    static ui8 ToInt(EColumn column) {
        auto columnInt = static_cast<ui8>(column);
#ifndef NDEBUG
        Y_VERIFY(columnInt < NumColumns);
#else
        Y_ASSUME(columnInt < NumColumns);
#endif
        return columnInt;
    }

public:
    constexpr bool HasColumn(EColumn column) const {
        return Mask_.test(ToInt(column));
    }

    constexpr TPointColumns WithColumn(EColumn column, bool flag = true) const {
        return TPointColumns{TBitSet{Mask_}.set(ToInt(column), flag)};
    }

    constexpr void SetColumn(EColumn column, bool flag = true) {
        Mask_.set(ToInt(column), flag);
    }

    constexpr bool HasStep() const {
        return Mask_.test(ToInt(Step));
    }

    constexpr TPointColumns WithStep(bool step = true) const {
        return TPointColumns{TBitSet{Mask_}.set(ToInt(Step), step)};
    }

    constexpr void SetStep(bool step = true) {
        Mask_.set(ToInt(Step), step);
    }

    constexpr bool HasMerge() const {
        return Mask_.test(ToInt(Merge));
    }

    constexpr TPointColumns WithMerge(bool merge = true) const {
        return TPointColumns{TBitSet{Mask_}.set(ToInt(Merge), merge)};
    }

    constexpr void SetMerge(bool merge = true) {
        Mask_.set(ToInt(Merge), merge);
    }

    constexpr bool HasCount() const {
        return Mask_.test(ToInt(Count));
    }

    constexpr TPointColumns WithCount(bool count = true) const {
        return TPointColumns{TBitSet{Mask_}.set(ToInt(Count), count)};
    }

    constexpr void SetCount(bool count = true) {
        Mask_.set(ToInt(Count), count);
    }

private:
    constexpr friend bool operator==(const TPointColumns& lhs, const TPointColumns& rhs) {
        return lhs.Mask_ == rhs.Mask_;
    }

    constexpr friend bool operator!=(const TPointColumns& lhs, const TPointColumns& rhs) {
        return lhs.Mask_ != rhs.Mask_;
    }

    inline constexpr friend TPointColumns operator|(TPointColumns lhs, TPointColumns rhs) {
        return lhs |= rhs;
    }

    inline constexpr friend TPointColumns& operator|=(TPointColumns& lhs, TPointColumns rhs) {
        return lhs.Mask_ |= rhs.Mask_, lhs;
    }

    inline constexpr friend TPointColumns operator&(TPointColumns lhs, TPointColumns rhs) {
        return lhs &= rhs;
    }

    inline constexpr friend TPointColumns& operator&=(TPointColumns& lhs, TPointColumns rhs) {
        return lhs.Mask_ &= rhs.Mask_, lhs;
    }

    inline constexpr friend TPointColumns operator^(TPointColumns lhs, TPointColumns rhs) {
        return lhs ^= rhs;
    }

    inline constexpr friend TPointColumns& operator^=(TPointColumns& lhs, TPointColumns rhs) {
        return lhs.Mask_ ^= rhs.Mask_, lhs;
    }

    // NB: not marked as constexpr as constant evaluation is not guaranteed by the standard
    inline /* constexpr */ friend TPointColumns operator~(TPointColumns ty) {
        return TPointColumns{~ty.Mask_};
    }

public:
    // XXX: this is a temporary method, do not use
    NTs::TColumnSet ToColumnSet(EPointType type) const;

private:
    TBitSet Mask_ = 0;
};

std::ostream& operator<<(std::ostream& os, TPointColumns type);

} // namespace NSolomon::NTsModel
