#pragma once

#include "iterator.h"
#include "points.h"

#include <solomon/libs/cpp/math/math.h>

namespace NSolomon::NTsModel {

/**
 * Point traits that only apply to scalar points.
 *
 * See `TPointTraits` below.
 */
template <typename TPoint_>
struct TScalarPointTraits {
    static_assert(IsPointV<TPoint_>, "expected a point");

    /**
     * Indicates that this point is not a scalar.
     */
    static const constexpr bool IsScalar = false;
};

template <>
struct TScalarPointTraits<TGaugePoint> {
    /**
     * Indicates that this point is a scalar.
     */
    static const constexpr bool IsScalar = true;

    /**
     * Compare points for equality. NaNs are considered equal.
     */
    static bool Eq(const TGaugePoint::TValue& lhs, const TGaugePoint::TValue& rhs) {
        return (std::isnan(lhs.Num) && std::isnan(rhs.Num)) || AreDoublesEqual(lhs.ValueDivided(), rhs.ValueDivided());
    }

    /**
     * Compare points for order. NaNs are compared according to usual rules.
     */
    static bool Lt(const TGaugePoint::TValue& lhs, const TGaugePoint::TValue& rhs) {
        return lhs.ValueDivided() < rhs.ValueDivided();
    }
};

template <>
struct TScalarPointTraits<TCounterPoint> {
    /**
     * Indicates that this point is a scalar.
     */
    static const constexpr bool IsScalar = true;

    /**
     * Compare points for equality.
     */
    static bool Eq(const TCounterPoint::TValue& lhs, const TCounterPoint::TValue& rhs) {
        return lhs.Value == rhs.Value;
    }

    /**
     * Compare points for order.
     */
    static bool Lt(const TCounterPoint::TValue& lhs, const TCounterPoint::TValue& rhs) {
        return lhs.Value < rhs.Value;
    }
};

template <>
struct TScalarPointTraits<TRatePoint> {
    /**
     * Indicates that this point is a scalar.
     */
    static const constexpr bool IsScalar = true;

    /**
     * Compare points for equality.
     */
    static bool Eq(const TRatePoint::TValue& lhs, const TRatePoint::TValue& rhs) {
        return lhs.Value == rhs.Value;
    }

    /**
     * Compare points for order.
     */
    static bool Lt(const TRatePoint::TValue& lhs, const TRatePoint::TValue& rhs) {
        return lhs.Value < rhs.Value;
    }
};

template <>
struct TScalarPointTraits<TIGaugePoint> {
    /**
     * Indicates that this point is a scalar.
     */
    static const constexpr bool IsScalar = true;

    /**
     * Compare points for equality.
     */
    static bool Eq(const TIGaugePoint::TValue& lhs, const TIGaugePoint::TValue& rhs) {
        return lhs.Value == rhs.Value;
    }

    /**
     * Compare points for order.
     */
    static bool Lt(const TIGaugePoint::TValue& lhs, const TIGaugePoint::TValue& rhs) {
        return lhs.Value < rhs.Value;
    }
};

/**
 * A convenience object which can be used to type-dispatch overloaded functions based on point type.
 *
 * See `visit.h`.
 */
template <typename TPoint_>
struct TPointTraits: public TScalarPointTraits<TPoint_> {
    static_assert(IsPointV<TPoint_>, "expected a point");

    using TPoint = TPoint_;

    /**
     * Get type of this point.
     */
    static EPointType Type() {
        return TPoint::Type;
    }

    /**
     * Create a default instance of this point.
     */
    static TPoint MakePoint() {
        return TPoint{};
    }

    /**
     * Take a generic iterator and cast it down to this point's type.
     */
    static IIterator<TPoint>& DowncastIterator(IGenericIterator& it) {
        return *it.As<TPoint>();
    }
    static const IIterator<TPoint>& DowncastIterator(const IGenericIterator& it) {
        return *it.As<TPoint>();
    }
    static IIterator<TPoint>* DowncastIterator(IGenericIterator* it) {
        return it->As<TPoint>();
    }
    static const IIterator<TPoint>* DowncastIterator(const IGenericIterator* it) {
        return it->As<TPoint>();
    }
    static THolder<IIterator<TPoint>> DowncastIterator(THolder<IGenericIterator> it) {
        return THolder(it.Release()->As<TPoint>());
    }
};

// XXX: this is a temporary trait, do not use
template <typename T>
struct TPointCoder {
    static_assert(TDependentFalse<T>, "not a point");
};

template <>
struct TPointCoder<TGaugePoint> {
    using TEncoder = NSolomon::NTs::TDoubleTsEncoder;
    using TDecoder = NSolomon::NTs::TDoubleTsDecoder;
};

template <>
struct TPointCoder<TCounterPoint> {
    using TEncoder = NSolomon::NTs::TCounterTsEncoder;
    using TDecoder = NSolomon::NTs::TCounterTsDecoder;
};

template <>
struct TPointCoder<TRatePoint> {
    using TEncoder = NSolomon::NTs::TCounterTsEncoder;
    using TDecoder = NSolomon::NTs::TCounterTsDecoder;
};

template <>
struct TPointCoder<TIGaugePoint> {
    using TEncoder = NSolomon::NTs::TGaugeIntTsEncoder;
    using TDecoder = NSolomon::NTs::TGaugeIntTsDecoder;
};

template <>
struct TPointCoder<THistPoint> {
    using TEncoder = NSolomon::NTs::THistogramTsEncoder;
    using TDecoder = NSolomon::NTs::THistogramTsDecoder;
};

template <>
struct TPointCoder<THistRatePoint> {
    using TEncoder = NSolomon::NTs::THistogramTsEncoder;
    using TDecoder = NSolomon::NTs::THistogramTsDecoder;
};

template <>
struct TPointCoder<TDSummaryPoint> {
    using TEncoder = NSolomon::NTs::TSummaryDoubleTsEncoder;
    using TDecoder = NSolomon::NTs::TSummaryDoubleTsDecoder;
};

template <>
struct TPointCoder<TLogHistPoint> {
    using TEncoder = NSolomon::NTs::TLogHistogramTsEncoder;
    using TDecoder = NSolomon::NTs::TLogHistogramTsDecoder;
};

template <typename T>
using TEncoder = typename TPointCoder<T>::TEncoder;
template <typename T>
using TDecoder = typename TPointCoder<T>::TDecoder;

} // namespace NSolomon::NTsModel
