#include <library/cpp/testing/gtest/gtest.h>

#include <solomon/libs/cpp/ts_model/aggregator_count.h>
#include <solomon/libs/cpp/ts_model/testlib/testlib.h>

using namespace NSolomon;
using namespace NSolomon::NTsModel;

template <typename T>
class CountAggregator: public testing::Test {
public:
    using TPoint = T;
};

TYPED_TEST_SUITE(CountAggregator, TPoints);

TYPED_TEST(CountAggregator, Basics) {
    TCountAggregator<typename TestFixture::TPoint> aggr;

    EXPECT_EQ(aggr.InputType(), TestFixture::TPoint::Type);
    EXPECT_EQ(aggr.OutputType(), TIGaugePoint::Type);
    EXPECT_EQ(aggr.Function(), EAggregationFunction::Count);
    EXPECT_EQ(aggr.Columns(), (TPointColumns{TPointColumns::Step, TPointColumns::Count}));
}

TYPED_TEST(CountAggregator, Empty) {
    TCountAggregator<typename TestFixture::TPoint> aggr;

    EXPECT_EQ(aggr.Count(), 0);
    EXPECT_FALSE(aggr.HasPoints());

    TIGaugePoint res = aggr.Finish();
    EXPECT_EQ(res.Value, 0);
    EXPECT_EQ(res.Time, TInstant::Zero());
    EXPECT_EQ(res.Step, TDuration::Zero());
    EXPECT_EQ(res.Count, 0u);
    EXPECT_EQ(res.Merge, false);
}

TYPED_TEST(CountAggregator, Some) {
    TCountAggregator<typename TestFixture::TPoint> aggr;

    EXPECT_EQ(aggr.Count(), 0);
    EXPECT_FALSE(aggr.HasPoints());

    typename TestFixture::TPoint point;

    point.Time = TInstant::ParseIso8601("2020-01-01T00:00:10Z");
    point.Step = TDuration::Seconds(15);
    aggr.Add(point);

    point.Time = TInstant::ParseIso8601("2020-01-01T00:00:20Z");
    point.Step = TDuration::Seconds(10);
    aggr.Add(point);

    point.Time = TInstant::ParseIso8601("2020-01-01T00:00:30Z");
    point.Step = TDuration::Seconds(20);
    aggr.Add(point);

    EXPECT_EQ(aggr.Count(), 3);
    EXPECT_TRUE(aggr.HasPoints());

    TIGaugePoint res = aggr.Finish();
    EXPECT_EQ(res.Value, 3);
    EXPECT_EQ(res.Time, TInstant::ParseIso8601("2020-01-01T00:00:10Z"));
    EXPECT_EQ(res.Step, TDuration::Seconds(15));
    EXPECT_EQ(res.Count, 3u);
    EXPECT_EQ(res.Merge, false);

    EXPECT_EQ(aggr.Count(), 0);
    EXPECT_FALSE(aggr.HasPoints());
}

TYPED_TEST(CountAggregator, SomeReset) {
    TCountAggregator<typename TestFixture::TPoint> aggr;

    EXPECT_EQ(aggr.Count(), 0);
    EXPECT_FALSE(aggr.HasPoints());

    typename TestFixture::TPoint point;

    point.Time = TInstant::ParseIso8601("2020-01-01T00:00:10Z");
    point.Step = TDuration::Seconds(15);
    aggr.Add(point);

    EXPECT_EQ(aggr.Count(), 1);
    EXPECT_TRUE(aggr.HasPoints());

    aggr.Reset();

    EXPECT_EQ(aggr.Count(), 0);
    EXPECT_FALSE(aggr.HasPoints());

    point.Time = TInstant::ParseIso8601("2020-01-01T00:00:20Z");
    point.Step = TDuration::Seconds(10);
    aggr.Add(point);

    point.Time = TInstant::ParseIso8601("2020-01-01T00:00:30Z");
    point.Step = TDuration::Seconds(20);
    aggr.Add(point);

    EXPECT_EQ(aggr.Count(), 2);
    EXPECT_TRUE(aggr.HasPoints());

    TIGaugePoint res = aggr.Finish();
    EXPECT_EQ(res.Value, 2);
    EXPECT_EQ(res.Time, TInstant::ParseIso8601("2020-01-01T00:00:20Z"));
    EXPECT_EQ(res.Step, TDuration::Seconds(10));
    EXPECT_EQ(res.Count, 2u);
    EXPECT_EQ(res.Merge, false);

    EXPECT_EQ(aggr.Count(), 0);
    EXPECT_FALSE(aggr.HasPoints());
}

TYPED_TEST(CountAggregator, AggrCount) {
    TCountAggregator<typename TestFixture::TPoint> aggr;

    EXPECT_EQ(aggr.Count(), 0);
    EXPECT_FALSE(aggr.HasPoints());

    typename TestFixture::TPoint point;

    point.Time = TInstant::ParseIso8601("2020-01-01T00:00:10Z");
    point.Step = TDuration::Seconds(15);
    point.Count = 2;
    aggr.Add(point);

    point.Time = TInstant::ParseIso8601("2020-01-01T00:00:20Z");
    point.Step = TDuration::Seconds(10);
    point.Count = 3;
    aggr.Add(point);

    EXPECT_EQ(aggr.Count(), 2);
    EXPECT_TRUE(aggr.HasPoints());

    TIGaugePoint res = aggr.Finish();
    EXPECT_EQ(res.Value, 2);
    EXPECT_EQ(res.Time, TInstant::ParseIso8601("2020-01-01T00:00:10Z"));
    EXPECT_EQ(res.Step, TDuration::Seconds(15));
    EXPECT_EQ(res.Count, 5u);
    EXPECT_EQ(res.Merge, false);

    EXPECT_EQ(aggr.Count(), 0);
    EXPECT_FALSE(aggr.HasPoints());
}

TEST(CountAggregatorEdgeCases, NanValues) {
    // TODO: make sure we're compatible with java implementation
}

TEST(CountAggregatorEdgeCases, NanValuesAggrCount) {
    // TODO: make sure we're compatible with java implementation
}
