#include <library/cpp/testing/gtest/gtest.h>

#include <solomon/libs/cpp/ts_model/aggregator_min.h>
#include <solomon/libs/cpp/ts_model/testlib/testlib.h>

using namespace NSolomon;
using namespace NSolomon::NTsModel;

template <typename T>
class MinAggregator: public testing::Test {
public:
    using TPoint = T;
};

TYPED_TEST_SUITE(MinAggregator, TScalarPoints);

TYPED_TEST(MinAggregator, Basics) {
    TMinAggregator<typename TestFixture::TPoint> aggr;

    EXPECT_EQ(aggr.InputType(), TestFixture::TPoint::Type);
    EXPECT_EQ(aggr.OutputType(), TestFixture::TPoint::Type);
    EXPECT_EQ(aggr.Function(), EAggregationFunction::Min);
    EXPECT_EQ(aggr.Columns(), (TPointColumns{TPointColumns::Step, TPointColumns::Count}));
}

TYPED_TEST(MinAggregator, Empty) {
    TMinAggregator<typename TestFixture::TPoint> aggr;

    EXPECT_EQ(aggr.Count(), 0);
    EXPECT_FALSE(aggr.HasPoints());

    auto res = aggr.Finish();
    EXPECT_EQ(ScalarValue(res), 0);
    EXPECT_EQ(res.Time, TInstant::Zero());
    EXPECT_EQ(res.Step, TDuration::Zero());
    EXPECT_EQ(res.Count, 0u);
    EXPECT_EQ(res.Merge, false);
}

TYPED_TEST(MinAggregator, Some) {
    TMinAggregator<typename TestFixture::TPoint> aggr;

    EXPECT_EQ(aggr.Count(), 0);
    EXPECT_FALSE(aggr.HasPoints());

    typename TestFixture::TPoint point;

    SetScalarValue(&point, 10);
    point.Time = TInstant::ParseIso8601("2020-01-01T00:00:10Z");
    point.Step = TDuration::Seconds(10);
    aggr.Add(point);

    SetScalarValue(&point, 20);
    point.Time = TInstant::ParseIso8601("2020-01-01T00:00:20Z");
    point.Step = TDuration::Seconds(15);
    aggr.Add(point);

    SetScalarValue(&point, 5);
    point.Time = TInstant::ParseIso8601("2020-01-01T00:00:30Z");
    point.Step = TDuration::Seconds(5);
    aggr.Add(point);

    SetScalarValue(&point, 5);
    point.Time = TInstant::ParseIso8601("2020-01-01T00:00:40Z");
    point.Step = TDuration::Seconds(15);
    aggr.Add(point);

    EXPECT_EQ(aggr.Count(), 4);
    EXPECT_TRUE(aggr.HasPoints());

    auto res = aggr.Finish();
    EXPECT_EQ(ScalarValue(res), 5);
    EXPECT_EQ(res.Time, TInstant::ParseIso8601("2020-01-01T00:00:30Z"));
    EXPECT_EQ(res.Step, TDuration::Seconds(5));
    EXPECT_EQ(res.Count, 4u);
    EXPECT_EQ(res.Merge, false);

    EXPECT_EQ(aggr.Count(), 0);
    EXPECT_FALSE(aggr.HasPoints());
}

TYPED_TEST(MinAggregator, MinBegin) {
    TMinAggregator<typename TestFixture::TPoint> aggr;

    EXPECT_EQ(aggr.Count(), 0);
    EXPECT_FALSE(aggr.HasPoints());

    typename TestFixture::TPoint point;

    SetScalarValue(&point, 10);
    point.Time = TInstant::ParseIso8601("2020-01-01T00:00:10Z");
    point.Step = TDuration::Seconds(10);
    aggr.Add(point);

    SetScalarValue(&point, 20);
    point.Time = TInstant::ParseIso8601("2020-01-01T00:00:20Z");
    point.Step = TDuration::Seconds(15);
    aggr.Add(point);

    EXPECT_EQ(aggr.Count(), 2);
    EXPECT_TRUE(aggr.HasPoints());

    auto res = aggr.Finish();
    EXPECT_EQ(ScalarValue(res), 10);
    EXPECT_EQ(res.Time, TInstant::ParseIso8601("2020-01-01T00:00:10Z"));
    EXPECT_EQ(res.Step, TDuration::Seconds(10));
    EXPECT_EQ(res.Count, 2u);
    EXPECT_EQ(res.Merge, false);
}

TYPED_TEST(MinAggregator, MinMiddle) {
    TMinAggregator<typename TestFixture::TPoint> aggr;

    EXPECT_EQ(aggr.Count(), 0);
    EXPECT_FALSE(aggr.HasPoints());

    typename TestFixture::TPoint point;

    SetScalarValue(&point, 10);
    point.Time = TInstant::ParseIso8601("2020-01-01T00:00:10Z");
    point.Step = TDuration::Seconds(10);
    aggr.Add(point);

    SetScalarValue(&point, 5);
    point.Time = TInstant::ParseIso8601("2020-01-01T00:00:20Z");
    point.Step = TDuration::Seconds(10);
    aggr.Add(point);

    SetScalarValue(&point, 20);
    point.Time = TInstant::ParseIso8601("2020-01-01T00:00:30Z");
    point.Step = TDuration::Seconds(15);
    aggr.Add(point);

    EXPECT_EQ(aggr.Count(), 3);
    EXPECT_TRUE(aggr.HasPoints());

    auto res = aggr.Finish();
    EXPECT_EQ(ScalarValue(res), 5);
    EXPECT_EQ(res.Time, TInstant::ParseIso8601("2020-01-01T00:00:20Z"));
    EXPECT_EQ(res.Step, TDuration::Seconds(10));
    EXPECT_EQ(res.Count, 3u);
    EXPECT_EQ(res.Merge, false);
}

TYPED_TEST(MinAggregator, MinEnd) {
    TMinAggregator<typename TestFixture::TPoint> aggr;

    EXPECT_EQ(aggr.Count(), 0);
    EXPECT_FALSE(aggr.HasPoints());

    typename TestFixture::TPoint point;

    SetScalarValue(&point, 20);
    point.Time = TInstant::ParseIso8601("2020-01-01T00:00:20Z");
    point.Step = TDuration::Seconds(15);
    aggr.Add(point);

    SetScalarValue(&point, 10);
    point.Time = TInstant::ParseIso8601("2020-01-01T00:00:10Z");
    point.Step = TDuration::Seconds(10);
    aggr.Add(point);

    EXPECT_EQ(aggr.Count(), 2);
    EXPECT_TRUE(aggr.HasPoints());

    auto res = aggr.Finish();
    EXPECT_EQ(ScalarValue(res), 10);
    EXPECT_EQ(res.Time, TInstant::ParseIso8601("2020-01-01T00:00:10Z"));
    EXPECT_EQ(res.Step, TDuration::Seconds(10));
    EXPECT_EQ(res.Count, 2u);
    EXPECT_EQ(res.Merge, false);
}

TYPED_TEST(MinAggregator, Reset) {
    TMinAggregator<typename TestFixture::TPoint> aggr;

    EXPECT_EQ(aggr.Count(), 0);
    EXPECT_FALSE(aggr.HasPoints());

    typename TestFixture::TPoint point;

    SetScalarValue(&point, -10);
    point.Time = TInstant::ParseIso8601("2020-01-01T00:00:10Z");
    point.Step = TDuration::Seconds(15);
    aggr.Add(point);

    EXPECT_EQ(aggr.Count(), 1);
    EXPECT_TRUE(aggr.HasPoints());

    aggr.Reset();

    EXPECT_EQ(aggr.Count(), 0);
    EXPECT_FALSE(aggr.HasPoints());

    SetScalarValue(&point, 5);
    point.Time = TInstant::ParseIso8601("2020-01-01T00:00:20Z");
    point.Step = TDuration::Seconds(10);
    aggr.Add(point);

    SetScalarValue(&point, 2);
    point.Time = TInstant::ParseIso8601("2020-01-01T00:00:30Z");
    point.Step = TDuration::Seconds(10);
    aggr.Add(point);

    EXPECT_EQ(aggr.Count(), 2);
    EXPECT_TRUE(aggr.HasPoints());

    auto res = aggr.Finish();
    EXPECT_EQ(ScalarValue(res), 2);
    EXPECT_EQ(res.Time, TInstant::ParseIso8601("2020-01-01T00:00:30Z"));
    EXPECT_EQ(res.Step, TDuration::Seconds(10));
    EXPECT_EQ(res.Count, 2u);
    EXPECT_EQ(res.Merge, false);

    EXPECT_EQ(aggr.Count(), 0);
    EXPECT_FALSE(aggr.HasPoints());
}

TYPED_TEST(MinAggregator, AggrCount) {
    TMinAggregator<typename TestFixture::TPoint> aggr;

    EXPECT_EQ(aggr.Count(), 0);
    EXPECT_FALSE(aggr.HasPoints());

    typename TestFixture::TPoint point;

    SetScalarValue(&point, 1);
    point.Time = TInstant::ParseIso8601("2020-01-01T00:00:10Z");
    point.Step = TDuration::Seconds(15);
    point.Count = 2;
    aggr.Add(point);

    SetScalarValue(&point, 1);
    point.Time = TInstant::ParseIso8601("2020-01-01T00:00:20Z");
    point.Step = TDuration::Seconds(10);
    point.Count = 3;
    aggr.Add(point);

    EXPECT_EQ(aggr.Count(), 2);
    EXPECT_TRUE(aggr.HasPoints());

    auto res = aggr.Finish();
    EXPECT_EQ(ScalarValue(res), 1);
    EXPECT_EQ(res.Time, TInstant::ParseIso8601("2020-01-01T00:00:10Z"));
    EXPECT_EQ(res.Step, TDuration::Seconds(15));
    EXPECT_EQ(res.Count, 5u);
    EXPECT_EQ(res.Merge, false);

    EXPECT_EQ(aggr.Count(), 0);
    EXPECT_FALSE(aggr.HasPoints());
}

TEST(MinAggregatorEdgeCases, NanValues) {
    // TODO: make sure we're compatible with java implementation
}

TEST(MinAggregatorEdgeCases, NanValuesAggrCount) {
    // TODO: make sure we're compatible with java implementation
}
