#include <library/cpp/testing/gtest/gtest.h>

#include <solomon/libs/cpp/ts_model/aggregator_sum.h>
#include <solomon/libs/cpp/ts_model/testlib/testlib.h>

using namespace NSolomon;
using namespace NSolomon::NTsModel;

template <typename T>
class SumAggregator: public testing::Test {
public:
    using TPoint = T;
};

TYPED_TEST_SUITE(SumAggregator, TPoints);

TYPED_TEST(SumAggregator, Basics) {
    TSumAggregator<typename TestFixture::TPoint> aggr;

    EXPECT_EQ(aggr.InputType(), TestFixture::TPoint::Type);
    EXPECT_EQ(aggr.OutputType(), TestFixture::TPoint::Type);
    EXPECT_EQ(aggr.Function(), EAggregationFunction::Sum);
    EXPECT_EQ(aggr.Columns(), (TPointColumns{TPointColumns::Step, TPointColumns::Count}));
}

template <typename T>
class SumAggregatorScalar: public testing::Test {
public:
    using TPoint = T;
};

TYPED_TEST_SUITE(SumAggregatorScalar, TScalarPoints);

TYPED_TEST(SumAggregatorScalar, Empty) {
    TSumAggregator<typename TestFixture::TPoint> aggr;

    EXPECT_EQ(aggr.Count(), 0);
    EXPECT_FALSE(aggr.HasPoints());

    auto res = aggr.Finish();
    EXPECT_EQ(ScalarValue(res), 0);
    EXPECT_EQ(res.Time, TInstant::Zero());
    EXPECT_EQ(res.Step, TDuration::Zero());
    EXPECT_EQ(res.Count, 0u);
    EXPECT_EQ(res.Merge, false);
}

TYPED_TEST(SumAggregatorScalar, Some) {
    TSumAggregator<typename TestFixture::TPoint> aggr;

    EXPECT_EQ(aggr.Count(), 0);
    EXPECT_FALSE(aggr.HasPoints());

    typename TestFixture::TPoint point;

    SetScalarValue(&point, 10);
    point.Time = TInstant::ParseIso8601("2020-01-01T00:00:10Z");
    point.Step = TDuration::Seconds(10);
    aggr.Add(point);

    SetScalarValue(&point, 20);
    point.Time = TInstant::ParseIso8601("2020-01-01T00:00:20Z");
    point.Step = TDuration::Seconds(15);
    aggr.Add(point);

    SetScalarValue(&point, 5);
    point.Time = TInstant::ParseIso8601("2020-01-01T00:00:30Z");
    point.Step = TDuration::Seconds(5);
    aggr.Add(point);

    EXPECT_EQ(aggr.Count(), 3);
    EXPECT_TRUE(aggr.HasPoints());

    auto res = aggr.Finish();
    EXPECT_EQ(ScalarValue(res), 35);
    EXPECT_EQ(res.Time, TInstant::ParseIso8601("2020-01-01T00:00:10Z"));
    EXPECT_EQ(res.Step, TDuration::Seconds(10));
    EXPECT_EQ(res.Count, 3u);
    EXPECT_EQ(res.Merge, false);

    EXPECT_EQ(aggr.Count(), 0);
    EXPECT_FALSE(aggr.HasPoints());
}

TYPED_TEST(SumAggregatorScalar, SomeReset) {
    TSumAggregator<typename TestFixture::TPoint> aggr;

    EXPECT_EQ(aggr.Count(), 0);
    EXPECT_FALSE(aggr.HasPoints());

    typename TestFixture::TPoint point;

    SetScalarValue(&point, -1);
    point.Time = TInstant::ParseIso8601("2020-01-01T00:00:10Z");
    point.Step = TDuration::Seconds(15);
    aggr.Add(point);

    EXPECT_EQ(aggr.Count(), 1);
    EXPECT_TRUE(aggr.HasPoints());

    aggr.Reset();

    EXPECT_EQ(aggr.Count(), 0);
    EXPECT_FALSE(aggr.HasPoints());

    SetScalarValue(&point, 5);
    point.Time = TInstant::ParseIso8601("2020-01-01T00:00:20Z");
    point.Step = TDuration::Seconds(10);
    aggr.Add(point);

    SetScalarValue(&point, -2);
    point.Time = TInstant::ParseIso8601("2020-01-01T00:00:30Z");
    point.Step = TDuration::Seconds(20);
    aggr.Add(point);

    EXPECT_EQ(aggr.Count(), 2);
    EXPECT_TRUE(aggr.HasPoints());

    auto res = aggr.Finish();
    EXPECT_EQ(ScalarValue(res), 3);
    EXPECT_EQ(res.Time, TInstant::ParseIso8601("2020-01-01T00:00:20Z"));
    EXPECT_EQ(res.Step, TDuration::Seconds(10));
    EXPECT_EQ(res.Count, 2u);
    EXPECT_EQ(res.Merge, false);

    EXPECT_EQ(aggr.Count(), 0);
    EXPECT_FALSE(aggr.HasPoints());
}

TYPED_TEST(SumAggregatorScalar, AggrCount) {
    TSumAggregator<typename TestFixture::TPoint> aggr;

    EXPECT_EQ(aggr.Count(), 0);
    EXPECT_FALSE(aggr.HasPoints());

    typename TestFixture::TPoint point;

    SetScalarValue(&point, 5);
    point.Time = TInstant::ParseIso8601("2020-01-01T00:00:10Z");
    point.Step = TDuration::Seconds(15);
    point.Count = 2;
    aggr.Add(point);

    SetScalarValue(&point, 10);
    point.Time = TInstant::ParseIso8601("2020-01-01T00:00:20Z");
    point.Step = TDuration::Seconds(10);
    point.Count = 3;
    aggr.Add(point);

    EXPECT_EQ(aggr.Count(), 2);
    EXPECT_TRUE(aggr.HasPoints());

    auto res = aggr.Finish();
    EXPECT_EQ(ScalarValue(res), 15);
    EXPECT_EQ(res.Time, TInstant::ParseIso8601("2020-01-01T00:00:10Z"));
    EXPECT_EQ(res.Step, TDuration::Seconds(15));
    EXPECT_EQ(res.Count, 5u);
    EXPECT_EQ(res.Merge, false);

    EXPECT_EQ(aggr.Count(), 0);
    EXPECT_FALSE(aggr.HasPoints());
}

TEST(SumAggregatorEdgeCases, NanValues) {
    // TODO: make sure we're compatible with java implementation
}

TEST(SumAggregatorEdgeCases, NanValuesAggrCount) {
    // TODO: make sure we're compatible with java implementation
}
