#include <library/cpp/testing/gtest/gtest.h>

#include <solomon/libs/cpp/ts_model/iterator_merge.h>
#include <solomon/libs/cpp/ts_model/iterator_list.h>
#include <solomon/libs/cpp/ts_model/testlib/testlib.h>

using namespace NSolomon::NTsModel;
using namespace NSolomon::NTs;

TEST(IteratorMerge, Empty) {
    TMergeIterator<TListIterator<TGaugePoint>> it{};

    ExpectIteratorsEq(std::move(it), TListIterator<TGaugePoint>{{}});
}

TEST(IteratorMerge, Consequent) {
    TVector<TMergeIterator<TListIterator<TGaugePoint>>::TWindow> points;

    points.push_back({
        TListIterator<TGaugePoint>{{
            Gauge("2020-01-01T00:00:00Z", 1, 0),
            Gauge("2020-01-01T00:00:10Z", 2, 0)
        }},
        TInstant::ParseIso8601("2020-01-01T00:00:00Z"),
        TInstant::ParseIso8601("2020-01-01T00:00:10Z")
    });
    points.push_back({
        TListIterator<TGaugePoint>{{
            Gauge("2020-01-01T00:00:20Z", 3, 0),
        }},
        TInstant::ParseIso8601("2020-01-01T00:00:20Z"),
        TInstant::ParseIso8601("2020-01-01T00:00:20Z")
    });
    points.push_back({
        TListIterator<TGaugePoint>{{
            Gauge("2020-01-01T00:00:30Z", 4, 0),
            Gauge("2020-01-01T00:00:40Z", 5, 0),
            Gauge("2020-01-01T00:00:50Z", 6, 0),
        }},
        TInstant::ParseIso8601("2020-01-01T00:00:30Z"),
        TInstant::ParseIso8601("2020-01-01T00:00:50Z")
    });

    TMergeIterator<TListIterator<TGaugePoint>> it{points.begin(), points.end()};

    ExpectIteratorsEq(
        std::move(it),
        TListIterator<TGaugePoint>{{
            Gauge("2020-01-01T00:00:00Z", 1, 0),
            Gauge("2020-01-01T00:00:10Z", 2, 0),
            Gauge("2020-01-01T00:00:20Z", 3, 0),
            Gauge("2020-01-01T00:00:30Z", 4, 0),
            Gauge("2020-01-01T00:00:40Z", 5, 0),
            Gauge("2020-01-01T00:00:50Z", 6, 0)
        }});
}

TEST(IteratorMerge, Split) {
    TVector<TMergeIterator<TListIterator<TGaugePoint>>::TWindow> points;

    points.push_back({
        TListIterator<TGaugePoint>{{
            Gauge("2020-01-01T00:00:00Z", 1, 0),
            Gauge("2020-01-01T00:00:10Z", 2, 0)
        }},
        TInstant::ParseIso8601("2020-01-01T00:00:00Z"),
        TInstant::ParseIso8601("2020-01-01T00:00:10Z")
    });
    points.push_back({
        TListIterator<TGaugePoint>{{
            Gauge("2020-01-01T00:00:20Z", 3, 0),
        }},
        TInstant::ParseIso8601("2020-01-01T00:00:20Z"),
        TInstant::ParseIso8601("2020-01-01T00:00:20Z")
    });
    points.push_back({
        TListIterator<TGaugePoint>{{
            Gauge("2020-01-01T00:00:05Z", 4, 0),
            Gauge("2020-01-01T00:00:15Z", 5, 0),
            Gauge("2020-01-01T00:00:25Z", 6, 0),
        }},
        TInstant::ParseIso8601("2020-01-01T00:00:05Z"),
        TInstant::ParseIso8601("2020-01-01T00:00:25Z")
    });

    TMergeIterator<TListIterator<TGaugePoint>> it{points.begin(), points.end()};

    ExpectIteratorsEq(
        std::move(it),
        TListIterator<TGaugePoint>{{
            Gauge("2020-01-01T00:00:00Z", 1, 0),
            Gauge("2020-01-01T00:00:05Z", 4, 0),
            Gauge("2020-01-01T00:00:10Z", 2, 0),
            Gauge("2020-01-01T00:00:15Z", 5, 0),
            Gauge("2020-01-01T00:00:20Z", 3, 0),
            Gauge("2020-01-01T00:00:25Z", 6, 0)
        }});
}

TEST(IteratorMerge, Merge) {
    TVector<TMergeIterator<TListIterator<TGaugePoint>>::TWindow> points;

    points.push_back({
        TListIterator<TGaugePoint>{{
            Gauge("2020-01-01T00:00:10Z", 1, 0),
            Gauge("2020-01-01T00:00:20Z", 2, 0, true, 1),
            Gauge("2020-01-01T00:00:30Z", 3, 0)
        }},
        TInstant::ParseIso8601("2020-01-01T00:00:10Z"),
        TInstant::ParseIso8601("2020-01-01T00:00:30Z")
    });
    points.push_back({
        TListIterator<TGaugePoint>{{
            Gauge("2020-01-01T00:00:50Z", 4, 0)
        }},
        TInstant::ParseIso8601("2020-01-01T00:00:50Z"),
        TInstant::ParseIso8601("2020-01-01T00:00:50Z")
    });
    points.push_back({
        TListIterator<TGaugePoint>{{
            Gauge("2020-01-01T00:00:10Z", 5, 0),
            Gauge("2020-01-01T00:00:20Z", 6, 0, true, 1)
        }},
        TInstant::ParseIso8601("2020-01-01T00:00:10Z"),
        TInstant::ParseIso8601("2020-01-01T00:00:20Z")
    });
    points.push_back({
        TListIterator<TGaugePoint>{{
            Gauge("2020-01-01T00:00:20Z", 7, 0, true, 1),
            Gauge("2020-01-01T00:00:30Z", 8, 0)
        }},
        TInstant::ParseIso8601("2020-01-01T00:00:20Z"),
        TInstant::ParseIso8601("2020-01-01T00:00:30Z")
    });
    points.push_back({
        TListIterator<TGaugePoint>{{
            Gauge("2020-01-01T00:00:40Z", 9, 0)
        }},
        TInstant::ParseIso8601("2020-01-01T00:00:40Z"),
        TInstant::ParseIso8601("2020-01-01T00:00:40Z")
    });

    TMergeIterator<TListIterator<TGaugePoint>> it{points.begin(), points.end()};

    ExpectIteratorsEq(
        std::move(it),
        TListIterator<TGaugePoint>{{
            Gauge("2020-01-01T00:00:10Z", 5, 0),
            Gauge("2020-01-01T00:00:20Z", 15, 0, true, 3),
            Gauge("2020-01-01T00:00:30Z", 8, 0),
            Gauge("2020-01-01T00:00:40Z", 9, 0),
            Gauge("2020-01-01T00:00:50Z", 4, 0)
        }});
}

TEST(IteratorMerge, EmptyIterators) {
    TVector<TMergeIterator<TListIterator<TGaugePoint>>::TWindow> points;

    points.push_back({
        TListIterator<TGaugePoint>{{
            Gauge("2020-01-01T00:00:00Z", 1, 0),
            Gauge("2020-01-01T00:00:10Z", 2, 0)
        }},
        TInstant::ParseIso8601("2020-01-01T00:00:00Z"),
        TInstant::ParseIso8601("2020-01-01T00:00:10Z")
    });
    points.push_back({
        TListIterator<TGaugePoint>{{
        }},
        TInstant::Zero(),
        TInstant::Max()
    });
    points.push_back({
        TListIterator<TGaugePoint>{{
            Gauge("2020-01-01T00:00:20Z", 3, 0),
        }},
        TInstant::ParseIso8601("2020-01-01T00:00:20Z"),
        TInstant::ParseIso8601("2020-01-01T00:00:20Z")
    });

    TMergeIterator<TListIterator<TGaugePoint>> it{points.begin(), points.end()};

    ExpectIteratorsEq(
        std::move(it),
        TListIterator<TGaugePoint>{{
            Gauge("2020-01-01T00:00:00Z", 1, 0),
            Gauge("2020-01-01T00:00:10Z", 2, 0),
            Gauge("2020-01-01T00:00:20Z", 3, 0)
        }});
}
