#include <library/cpp/testing/gtest/gtest.h>

#include <solomon/libs/cpp/ts_model/merge.h>

using namespace NSolomon::NTsModel;
using namespace NSolomon::NTs;

#define EXPECT_MERGE_EQ(left, right, expected) \
    do {                                       \
        auto l = (left);                       \
        auto r = (right);                      \
                                               \
        ::NSolomon::NTsModel::Merge(l, &r);    \
                                               \
        EXPECT_EQ(r, (expected));              \
    } while (false)

template <typename T>
class MergeValuePoint: public testing::Test {
public:
    T MakePoint(double num, ui64 denom = 0, ui64 count = 0, bool merge = true) {
        T point{};
        point.Num = num;
        point.Denom = denom;
        point.Count = count;
        point.Merge = merge;
        return point;
    }
};

using ValuePoints = ::testing::Types<TGaugePoint>;
TYPED_TEST_SUITE(MergeValuePoint, ValuePoints);

TYPED_TEST(MergeValuePoint, MoMerge) {
    EXPECT_MERGE_EQ(this->MakePoint(10, 0, 2), this->MakePoint(5, 0, 2, false), this->MakePoint(5, 0, 2, false));
}

TYPED_TEST(MergeValuePoint, ZeroDenom) {
    EXPECT_MERGE_EQ(this->MakePoint(10, 0, 2), this->MakePoint(5, 0, 1), this->MakePoint(15, 0, 3));
}

TYPED_TEST(MergeValuePoint, SameDenom) {
    EXPECT_MERGE_EQ(this->MakePoint(10, 3, 2), this->MakePoint(5, 3, 1), this->MakePoint(15, 3, 3));
}

TYPED_TEST(MergeValuePoint, DifferentDenom) {
    EXPECT_MERGE_EQ(this->MakePoint(10, 2, 1), this->MakePoint(5, 5, 1), this->MakePoint(6000, 0, 2));
}

TYPED_TEST(MergeValuePoint, LeftNan) {
    // TODO: do we add counts when one of points is nan?
    EXPECT_MERGE_EQ(this->MakePoint(NAN, 0, 2), this->MakePoint(5, 0, 1), this->MakePoint(5, 0, 3));
}

TYPED_TEST(MergeValuePoint, RightNan) {
    // TODO: do we add counts when one of points is nan?
    EXPECT_MERGE_EQ(this->MakePoint(10, 0, 2), this->MakePoint(NAN, 0, 1), this->MakePoint(10, 0, 3));
}

// this test was copied from java code
TYPED_TEST(MergeValuePoint, Java) {
    EXPECT_MERGE_EQ(this->MakePoint(3, 0), this->MakePoint(2, 0), this->MakePoint(5, 0));
    EXPECT_MERGE_EQ(this->MakePoint(3, 1000), this->MakePoint(2, 1000), this->MakePoint(5, 1000));
    EXPECT_MERGE_EQ(this->MakePoint(3, 1000), this->MakePoint(4, 2000), this->MakePoint(5, 0));

    EXPECT_MERGE_EQ(this->MakePoint(NAN, 100), this->MakePoint(NAN, 200), this->MakePoint(NAN, 0));
    EXPECT_MERGE_EQ(this->MakePoint(NAN, 100), this->MakePoint(NAN, 100), this->MakePoint(NAN, 0));

    EXPECT_MERGE_EQ(this->MakePoint(NAN, 6000), this->MakePoint(300, 5000), this->MakePoint(300, 5000));
    EXPECT_MERGE_EQ(this->MakePoint(300, 5000), this->MakePoint(NAN, 6000), this->MakePoint(300, 5000));

    EXPECT_MERGE_EQ(this->MakePoint(300, 5000), this->MakePoint(NAN, 0), this->MakePoint(300, 5000));
    EXPECT_MERGE_EQ(this->MakePoint(NAN, 0), this->MakePoint(300, 5000), this->MakePoint(300, 5000));
}

template <typename T>
class MergeHist: public testing::Test {
public:
    T MakePoint(std::vector<typename T::TBucket> buckets, ui64 denom = 0, ui64 count = 0, bool merge = true) {
        T point{};
        point.Buckets = std::move(buckets);
        point.Denom = denom;
        point.Count = count;
        point.Merge = merge;
        return point;
    }
};

using Hists = ::testing::Types<THistPoint, THistRatePoint>;
TYPED_TEST_SUITE(MergeHist, Hists);

TYPED_TEST(MergeHist, NoMerge) {
    EXPECT_MERGE_EQ(
        this->MakePoint({{10, 1}, {30, 2}, {50, 3}}),
        this->MakePoint({{10, 5}, {30, 1}, {50, 4}}, 0, 1, false),
        this->MakePoint({{10, 5}, {30, 1}, {50, 4}}, 0, 1, false));
}

TYPED_TEST(MergeHist, LeftEmpty) {
    EXPECT_MERGE_EQ(
        this->MakePoint({}),
        this->MakePoint({{1, 0}, {5, 3}, {10, 0}}),
        this->MakePoint({{1, 0}, {5, 3}, {10, 0}}));
}

TYPED_TEST(MergeHist, RightEmpty) {
    EXPECT_MERGE_EQ(
        this->MakePoint({{1, 0}, {5, 3}, {10, 0}}),
        this->MakePoint({}),
        this->MakePoint({{1, 0}, {5, 3}, {10, 0}}));
}

TYPED_TEST(MergeHist, SameBounds) {
    EXPECT_MERGE_EQ(
        this->MakePoint({{10, 1}, {30, 2}, {50, 3}}),
        this->MakePoint({{10, 5}, {30, 1}, {50, 4}}),
        this->MakePoint({{10, 6}, {30, 3}, {50, 7}}));
}

TYPED_TEST(MergeHist, DifferentBoundsLeftSmaller) {
    EXPECT_MERGE_EQ(
        this->MakePoint({{10, 1}, {30, 2}, {50, 3}}),
        this->MakePoint({{10, 1}, {20, 2}, {30, 3}, {40, 4}, {50, 5}}),
        this->MakePoint({{10, 2}, {20, 2}, {30, 5}, {40, 4}, {50, 8}}));
}

TYPED_TEST(MergeHist, DifferentBoundsRightSmaller) {
    EXPECT_MERGE_EQ(
        this->MakePoint({{10, 1}, {20, 2}, {30, 3}, {40, 4}, {50, 5}, {70, 2}}),
        this->MakePoint({{10, 1}, {30, 2}, {50, 3}, {60, 1}}),
        this->MakePoint({{10, 2}, {30, 7}, {50, 12}, {60, 3}}));
}

TYPED_TEST(MergeHist, SameBoundsIncreaseDenom) {
    // TODO: why 125?
    EXPECT_MERGE_EQ(
        this->MakePoint({{10, 50}, {20, 0}, {30, 0}}, 5000),
        this->MakePoint({{10, 100}, {30, 0}, {50, 0}}, 10000),
        this->MakePoint({{10, 125}, {30, 0}, {50, 0}}, 10000));
}

TYPED_TEST(MergeHist, SameBoundsDecreaseDenom) {
    // TODO: why 250?
    EXPECT_MERGE_EQ(
        this->MakePoint({{10, 100}, {30, 0}, {50, 0}}, 10000),
        this->MakePoint({{10, 50}, {30, 0}, {50, 0}}, 5000),
        this->MakePoint({{10, 250}, {30, 0}, {50, 0}}, 5000));
}

TYPED_TEST(MergeHist, DifferentBoundsIncreaseDenom) {
    // TODO: why 125?
    EXPECT_MERGE_EQ(
        this->MakePoint({{10, 50}, {30, 0}, {50, 50}}, 5000),
        this->MakePoint({{10, 100}, {20, 0}, {30, 0}, {40, 0}, {50, 0}}, 10000),
        this->MakePoint({{10, 125}, {20, 0}, {30, 0}, {40, 0}, {50, 25}}, 10000));
}

TYPED_TEST(MergeHist, DifferentBoundsDecreaseDenom) {
    // TODO: why 250?
    EXPECT_MERGE_EQ(
        this->MakePoint({{10, 100}, {20, 0}, {30, 0}, {40, 0}, {50, 0}}, 10000),
        this->MakePoint({{10, 50}, {30, 0}, {50, 50}}, 5000),
        this->MakePoint({{10, 250}, {30, 0}, {50, 50}}, 5000));
}

class MergeSummary: public testing::Test {
public:
    TDSummaryPoint MakePoint(i64 countValue, double sum, double min, double max, double last, ui64 count = 0, bool merge = true) {
        TDSummaryPoint point{};
        point.CountValue = countValue;
        point.Sum = sum;
        point.Min = min;
        point.Max = max;
        point.Last = last;
        point.Count = count;
        point.Merge = merge;
        return point;
    }
};

TEST_F(MergeSummary, NoMerge) {
    EXPECT_MERGE_EQ(
        this->MakePoint(2, 2, 1, 1, 1, 2),
        this->MakePoint(3, 4, 1, 2, 2, 3, false),
        this->MakePoint(3, 4, 1, 2, 2, 3, false));
}

TEST_F(MergeSummary, LeftEmpty) {
    // TODO: do we add counts when one of points is empty?
    EXPECT_MERGE_EQ(
        this->MakePoint(0, 0, 0, 0, 0, 1),
        this->MakePoint(3, 4, 1, 2, 2, 3),
        this->MakePoint(3, 4, 1, 2, 2, 4));
}

TEST_F(MergeSummary, RightEmpty) {
    // TODO: do we add counts when one of points is empty?
    EXPECT_MERGE_EQ(
        this->MakePoint(3, 4, 1, 2, 2, 3),
        this->MakePoint(0, 0, 0, 0, 0, 1),
        this->MakePoint(3, 4, 1, 2, 2, 4));
}

TEST_F(MergeSummary, Merge) {
    EXPECT_MERGE_EQ(
        this->MakePoint(3, 4, 1, 2, 2, 3),
        this->MakePoint(2, 5, 2, 3, 3, 1),
        this->MakePoint(5, 9, 1, 3, 3, 4));
}

TEST_F(MergeSummary, MergeFew) {
    auto one = this->MakePoint(10, 100, 2, 50, 4);
    auto two = this->MakePoint(5, 25, 1, 10, 8);
    auto three = this->MakePoint(15, 50, 4, 8, 5.5);
    auto expected = this->MakePoint(30, 175, 1, 50, 5.5);

    Merge(one, &two);
    Merge(two, &three);

    EXPECT_EQ(three, expected);
}

class MergeLogHist: public testing::Test {
public:
    TLogHistPoint MakePoint(ui64 zeroCount, i16 startPower, double Base, std::vector<double> values,
                            i16 maxBucketCount, ui64 count = 0, bool merge = true)
    {
        TLogHistPoint point{};
        point.ZeroCount = zeroCount;
        point.StartPower = startPower;
        point.Base = Base;
        point.Values = std::move(values);
        point.MaxBucketCount = maxBucketCount;
        point.Count = count;
        point.Merge = merge;
        return point;
    }
};

TEST_F(MergeLogHist, NoMerge) {
    EXPECT_MERGE_EQ(
        this->MakePoint(0, 0, 1.5, {1, 2, 3}, 100, 2),
        this->MakePoint(0, 0, 1.5, {1, 2, 3}, 100, 3, false),
        this->MakePoint(0, 0, 1.5, {1, 2, 3}, 100, 3, false));
}

TEST_F(MergeLogHist, LeftEmpty) {
    EXPECT_MERGE_EQ(
        this->MakePoint(1, 0, 1.5, {}, 100, 2),
        this->MakePoint(2, 0, 1.5, {1, 2, 3}, 100, 3),
        this->MakePoint(3, 0, 1.5, {1, 2, 3}, 100, 5));
}

TEST_F(MergeLogHist, RightEmpty) {
    EXPECT_MERGE_EQ(
        this->MakePoint(1, 0, 1.5, {1, 2, 3}, 100, 3),
        this->MakePoint(2, 0, 1.5, {}, 100, 2),
        this->MakePoint(3, 0, 1.5, {1, 2, 3}, 100, 5));
}

TEST_F(MergeLogHist, SameBounds) {
    EXPECT_MERGE_EQ(
        this->MakePoint(1, 0, 1.5, {1, 2, 3}, 100, 2),
        this->MakePoint(2, 0, 1.5, {5, 1, 4}, 100, 3),
        this->MakePoint(3, 0, 1.5, {6, 3, 7}, 100, 5));
}

TEST_F(MergeLogHist, DifferentBoundsLeftSmaller) {
    EXPECT_MERGE_EQ(
        this->MakePoint(0, 0, 1.5, {4, 6, 27}, 100, 2),
        this->MakePoint(0, 0, 1.5, {20, 4, 2, 1, 3}, 100, 3),
        this->MakePoint(0, 0, 1.5, {24, 10, 29, 1, 3}, 100, 5));
}

TEST_F(MergeLogHist, DifferentBoundsRightSmaller) {
    EXPECT_MERGE_EQ(
        this->MakePoint(0, 0, 1.5, {20, 4, 2, 1, 3}, 100, 3),
        this->MakePoint(0, 0, 1.5, {4, 6, 27}, 100, 2),
        this->MakePoint(0, 0, 1.5, {24, 10, 29, 1, 3}, 100, 5));
}

TEST_F(MergeLogHist, CountZero) {
    EXPECT_MERGE_EQ(
        this->MakePoint(1, 0, 1.5, {}, 100, 2),
        this->MakePoint(2, 0, 1.5, {}, 100, 3),
        this->MakePoint(3, 0, 1.5, {}, 100, 5));
}

TEST_F(MergeLogHist, WithDifferentStartPowerExtendUp) {
    EXPECT_MERGE_EQ(
        this->MakePoint(0, 1, 1.5, {1, 2, 3}, 100, 2),
        this->MakePoint(0, 2, 1.5, {10, 20, 30}, 100, 3),
        this->MakePoint(0, 1, 1.5, {1, 12, 23, 30}, 100, 5));
    EXPECT_MERGE_EQ(
        this->MakePoint(0, 2, 1.5, {10, 20, 30}, 100, 3),
        this->MakePoint(0, 1, 1.5, {1, 2, 3}, 100, 2),
        this->MakePoint(0, 1, 1.5, {1, 12, 23, 30}, 100, 5));
}

TEST_F(MergeLogHist, WithDifferentStartPowerExtendDown) {
    EXPECT_MERGE_EQ(
        this->MakePoint(0, 0, 1.5, {1, 2, 3}, 100, 2),
        this->MakePoint(0, -2, 1.5, {10, 20, 30}, 100, 3),
        this->MakePoint(0, -2, 1.5, {10, 20, 31, 2, 3}, 100, 5));
    EXPECT_MERGE_EQ(
        this->MakePoint(0, -2, 1.5, {10, 20, 30}, 100, 3),
        this->MakePoint(0, 0, 1.5, {1, 2, 3}, 100, 2),
        this->MakePoint(0, -2, 1.5, {10, 20, 31, 2, 3}, 100, 5));
}

TEST_F(MergeLogHist, WithDifferentStartPowerExtendUpAndDown) {
    EXPECT_MERGE_EQ(
        this->MakePoint(0, 0, 1.5, {1, 2, 3}, 100, 2),
        this->MakePoint(0, -1, 1.5, {10, 20, 30, 40, 50}, 100, 3),
        this->MakePoint(0, -1, 1.5, {10, 21, 32, 43, 50}, 100, 5));
    EXPECT_MERGE_EQ(
        this->MakePoint(0, -1, 1.5, {10, 20, 30, 40, 50}, 100, 3),
        this->MakePoint(0, 0, 1.5, {1, 2, 3}, 100, 2),
        this->MakePoint(0, -1, 1.5, {10, 21, 32, 43, 50}, 100, 5));
}

TEST_F(MergeLogHist, WithDifferentStartPowerExtendUpWithOverflow) {
    EXPECT_MERGE_EQ(
        this->MakePoint(0, 1, 1.5, {1, 2, 3, 4, 5, 6}, 100, 2),
        this->MakePoint(0, 0, 1.5, {10, 20, 30}, 5, 3),
        this->MakePoint(0, 2, 1.5, {10 + 21 + 32, 3, 4, 5, 6}, 5, 5));
}

TEST_F(MergeLogHist, WithDifferentStartPowerExtendDownWithOverflow) {
    EXPECT_MERGE_EQ(
        this->MakePoint(0, -3, 1.5, {1, 2, 3, 4, 5, 6}, 100, 2),
        this->MakePoint(0, 0, 1.5, {10, 20, 30}, 5, 3),
        this->MakePoint(0, -2, 1.5, {3, 3, 14, 25, 36}, 5, 5));
    EXPECT_MERGE_EQ(
        this->MakePoint(0, -3, 1.5, {1, 2, 3, 4, 5, 6}, 100, 2),
        this->MakePoint(0, 0, 1.5, {10, 20, 30, 40, 50}, 5, 3),
        this->MakePoint(0, 0, 1.5, {20, 25, 36, 40, 50}, 5, 5));
}

TEST_F(MergeLogHist, ChangedBaseZeroCount) {
    EXPECT_MERGE_EQ(
        this->MakePoint(1, 0, 1.5, {}, 100, 3),
        this->MakePoint(2, 0, 2, {}, 100, 2),
        this->MakePoint(3, 0, 2, {}, 100, 5));
}

TEST_F(MergeLogHist, ChangedBaseToHighValue) {
    EXPECT_MERGE_EQ(
        this->MakePoint(0, 0, 1.5, {10, 20, 30, 40, 50}, 100, 3),
        this->MakePoint(0, 0, 2, {1, 2, 3, 4}, 100, 2),
        this->MakePoint(0, 0, 2, {31, 72, 53, 4}, 100, 5));
}

TEST_F(MergeLogHist, ChangedBaseToLowValue) {
    EXPECT_MERGE_EQ(
        this->MakePoint(0, 0, 2, {10, 20, 30}, 100, 3),
        this->MakePoint(0, 0, 1.5, {1, 2, 3, 4, 5}, 100, 2),
        this->MakePoint(0, 0, 1.5, {11, 22, 3, 34, 5}, 100, 5));
}

TEST_F(MergeLogHist, ChangedBaseExtendUp) {
    EXPECT_MERGE_EQ(
        this->MakePoint(0, 0, 2, {10, 20, 30, 40, 50}, 100, 3),
        this->MakePoint(0, 0, 1.5, {1, 2, 3, 4, 5}, 100, 2),
        this->MakePoint(0, 0, 1.5, {11, 22, 3, 34, 5, 40, 50}, 100, 5));
}

TEST_F(MergeLogHist, ChangedBaseExtendUpWithOverflow) {
    EXPECT_MERGE_EQ(
        this->MakePoint(0, 0, 2, {10, 20, 30, 40, 50}, 100, 3),
        this->MakePoint(0, 0, 1.5, {1, 2, 3, 4, 5}, 5, 2),
        this->MakePoint(0, 2, 1.5, {11 + 22 + 3, 34, 5, 40, 50}, 5, 5));
}

TEST_F(MergeLogHist, ChangedBaseExtendDown) {
    EXPECT_MERGE_EQ(
        this->MakePoint(0, -2, 1.5, {0, 10, 20, 30, 40, 50}, 100, 3),
        this->MakePoint(0, 0, 2, {1, 2, 3, 4}, 100, 2),
        this->MakePoint(0, -1, 2, {10, 51, 92, 3, 4}, 100, 5));
    EXPECT_MERGE_EQ(
        this->MakePoint(0, -2, 1.5, {5, 10, 20, 30, 40, 50}, 100, 3),
        this->MakePoint(0, 0, 2, {1, 2, 3, 4}, 100, 2),
        this->MakePoint(0, -2, 2, {5, 10, 51, 92, 3, 4}, 100, 5));
}

TEST_F(MergeLogHist, ChangedBaseExtendDownWithOverflow) {
    EXPECT_MERGE_EQ(
        this->MakePoint(0, -2, 1.5, {5, 10, 20, 30, 40, 50}, 100, 3),
        this->MakePoint(0, 0, 2, {1, 2, 3, 4}, 5, 2),
        this->MakePoint(0, -1, 2, {5 + 10, 51, 92, 3, 4}, 5, 5));
}
