#include "dao_base.h"

#include <library/cpp/monlib/metrics/metric_registry.h>
#include <library/cpp/monlib/metrics/timer.h>

using namespace NMonitoring;
using namespace NThreading;
using namespace NYdb::NTable;


namespace NSolomon::NDb {
    TDaoBase::TDaoBase(std::shared_ptr<TTableClient> client, TMetricRegistry& registry)
        : Client_{std::move(client)}
        , Registry_{registry}
    {
    }

    void TDaoBase::Init() {
        RawQueries_ = LoadQueries();
        CreateCounters();
    }

    void TDaoBase::CreateCounters() {
        auto name = Name();

        for (auto method: Methods()) {
            Counters_.emplace(
                std::piecewise_construct,
                std::forward_as_tuple(method),
                std::forward_as_tuple(Registry_, name, method)
            );
        }
    }

    TCallScopeCounter TDaoBase::CounterFor(TStringBuf method) const {
        if (auto* c = Counters_.FindPtr(method)) {
            return TCallScopeCounter{*c};
        }

        // shouldn't really happen in runtime, since all possible args are known in compile time
        Y_FAIL("cannot find counter '%.*s'", static_cast<int>(method.size()), method.data());
    }

    template <typename T>
    auto TDaoBase::CreateComletionFunction(TCallScopeCounter counter) {
        return [c=std::move(counter)] (const T& f) -> TFuture<void> {
            auto&& v = f.GetValue();
            if (!v.IsSuccess()) {
                c.RecordError();
                ythrow yexception() << v.GetIssues().ToString();
            }
            return MakeFuture();
        };
    }

    TDaoBase::TCompletionFunction<NYdb::TAsyncStatus> TDaoBase::CompleteStatus(TCallScopeCounter counter) {
        return CreateComletionFunction<NYdb::TAsyncStatus>(std::move(counter));
    }

} // namespace NSolomon::NDb
