#pragma once

#include "util.h"
#include "counters.h"

#include <ydb/public/sdk/cpp/client/ydb_table/table.h>

#include <util/generic/hash.h>

#include <utility>

namespace NSolomon::NDb {
    class TDaoBase {
    protected:
        TDaoBase(std::shared_ptr<NYdb::NTable::TTableClient> client, NMonitoring::TMetricRegistry& registry);

        virtual THashMap<TStringBuf, TString> LoadQueries() = 0;
        virtual const TString& Name() const = 0;
        virtual const TVector<TStringBuf>& Methods() const = 0;

        void Init();
        TCallScopeCounter CounterFor(TStringBuf method) const;

        template <typename T>
        using TCompletionFunction = std::function<NThreading::TFuture<void>(T)>;

        TCompletionFunction<NYdb::TAsyncStatus> CompleteStatus(TCallScopeCounter);

        template <typename T, typename TParseFunc>
        static void ReadDataWithRetries(NYdb::NTable::TTableClient& client, const TString& query, NYdb::TParams params,
            NThreading::TPromise<T> p, TParseFunc parser, TCallScopeCounter counter)
        {
            auto handleResult = [=, c=std::move(counter)] (const NYdb::NTable::TAsyncDataQueryResult& f) mutable {
                if (f.HasException()) {
                    try {
                        f.GetValueSync();
                    } catch (...) {
                        c.RecordError();
                        p.SetException(std::current_exception());
                        throw;
                    }
                }

                auto&& v = f.GetValueSync();
                if (v.IsSuccess()) {
                    p.SetValue(parser(v));
                } else {
                    c.RecordError();
                }

                return NThreading::MakeFuture(NYdb::TStatus(v));
            };

            client.RetryOperation<NYdb::TStatus>([=, params{std::move(params)}] (NYdb::NTable::TSession s) mutable {
                return ExecutePrepared(std::move(s), query, params).Apply(handleResult);
            }).Subscribe([p] (auto f) mutable {
                try {
                    auto&& v = f.GetValue();
                    if (!v.IsSuccess()) {
                        p.SetException(v.GetIssues().ToString());
                    }
                } catch (...) {
                    p.SetException(std::current_exception());
                }
            });
        }

    protected:
        std::shared_ptr<NYdb::NTable::TTableClient> Client_;
        NMonitoring::TMetricRegistry& Registry_;
        THashMap<TStringBuf, TString> RawQueries_;

    private:
        void CreateCounters();

        template <typename T>
        auto CreateComletionFunction(TCallScopeCounter);
        THashMap<TString, TMethodCounter> Counters_;
    };
} // namespace NSolomon::NDb
