#include "driver.h"

#include <solomon/libs/cpp/cloud/iam/client.h>

#include <kikimr/public/sdk/cpp/client/iam/iam.h>
#include <kikimr/public/sdk/cpp/client/tvm/tvm.h>
#include <ydb/public/lib/jwt/jwt.h>

namespace NSolomon::NDb {
namespace {

using namespace NSolomon::NCloud;
using namespace NYdb;

constexpr auto DEFAULT_THREAD_COUNT = 2;
constexpr auto SCHEME_GRPCS = TStringBuf{"grpcs://"};
constexpr auto SCHEME_GRPC = TStringBuf{"grpc://"};

class TPrivateIamTokenProvider: public NYdb::ICredentialsProvider {
public:
    TPrivateIamTokenProvider(ITokenProviderPtr tokenProvider)
        : TokenProvider_{std::move(tokenProvider)}
    {
    }

private:
    TStringType GetAuthInfo() const override {
        if (auto container = TokenProvider_->Token()) {
            return TString{container->Token()};
        }

        return {};
    }

    bool IsValid() const override {
        return true;
    }

private:
    ITokenProviderPtr TokenProvider_;
};

class TPrivateIamTokenProviderFactory: public NYdb::ICredentialsProviderFactory {
public:
    TPrivateIamTokenProviderFactory(ITokenProviderPtr tokenProvider)
        : TokenProvider_{std::move(tokenProvider)}
    {}

private:
    TCredentialsProviderPtr CreateProvider() const override {
        return std::make_shared<TPrivateIamTokenProvider>(TokenProvider_);
    }

    TCredentialsProviderPtr CreateProvider(std::weak_ptr<ICoreFacility>) const override {
        return CreateProvider();
    }

    TStringType GetClientIdentity() const override {
        return "PRIVATE_API_IAM_PROVIDER" + ToString(reinterpret_cast<ui64>(this));
    }

private:
    ITokenProviderPtr TokenProvider_;
};

NYdb::EBalancingPolicy FromProtoPolicy(TYdbConfig::EBalancingPolicy policy) {
    switch (policy) {
        case TYdbConfig::ALL:
            return NYdb::EBalancingPolicy::UseAllNodes;
        case TYdbConfig::LOCAL:
            return NYdb::EBalancingPolicy::UsePreferableLocation;
        default:
            ythrow yexception() << "Unknown balancing policy: " << TYdbConfig::EBalancingPolicy_Name(policy);
    }
}

std::shared_ptr<NYdb::ICredentialsProviderFactory> CreateAuthProviderFactory(
    const NDb::TYdbConfig& conf,
    NSecrets::ISecretProvider& secretProvider,
    NSolomon::NCloud::ITokenProviderPtr iamTokenProviderPrivateApi)
{
    if (conf.has_tvm_auth()) {
        const auto& tvm = conf.tvm_auth();
        auto clientSecret = secretProvider.GetSecret(tvm.client_secret());
        Y_ENSURE(clientSecret.has_value(), "cannot find YDB TVM secret by key: " << tvm.client_secret());
        return NYdb::CreateTVMCredentialsProviderFactory(*clientSecret, tvm.client_id());
    }

    if (conf.has_iam_key_auth()) {
        const auto& iam = conf.iam_key_auth();
        return NYdb::CreateIamJwtFileCredentialsProviderFactory(
            {{.Endpoint = iam.iam_endpoint()}, iam.path()});
    }

    if (conf.has_private_iam_key_auth()) {
        return std::make_shared<TPrivateIamTokenProviderFactory>(std::move(iamTokenProviderPrivateApi));
    }

    return NYdb::CreateInsecureCredentialsProviderFactory();
}

} // namespace

NYdb::TDriver CreateDriver(
        const NDb::TYdbConfig& conf,
        NSecrets::ISecretProvider& secretProvider,
        NSolomon::NCloud::ITokenProviderPtr iamTokenProviderPrivateApi)
{
    NYdb::TDriverConfig driverConfig;
    driverConfig.SetDatabase(conf.GetDatabase());

    const TString& address = conf.GetAddress();
    if (address.StartsWith(SCHEME_GRPCS)) {
        driverConfig.SetEndpoint(address.substr(SCHEME_GRPCS.size()));
        driverConfig.UseSecureConnection();
    } else if (address.StartsWith(SCHEME_GRPC)) {
        driverConfig.SetEndpoint(address.substr(SCHEME_GRPC.size()));
    } else {
        driverConfig.SetEndpoint(address);
    }

    driverConfig.SetClientThreadsNum(conf.GetClientThreadCount() ? conf.GetClientThreadCount() : DEFAULT_THREAD_COUNT);
    driverConfig.SetBalancingPolicy(FromProtoPolicy(conf.GetBalancingPolicy()));
    driverConfig.SetCredentialsProviderFactory(
            CreateAuthProviderFactory(conf, secretProvider, std::move(iamTokenProviderPrivateApi)));
    return NYdb::TDriver{driverConfig};
}

} // namespace NSolomon::NDb
