#pragma once

#include "parser.h"
#include "util.h"
#include "counters.h"

#include <ydb/public/sdk/cpp/client/ydb_params/params.h>
#include <ydb/public/sdk/cpp/client/ydb_table/table.h>

#include <library/cpp/threading/future/future.h>

#include <util/generic/ptr.h>
#include <util/generic/vector.h>

#include <optional>

namespace NSolomon::NDb {

/// Helper class that reads whole table page by page and concatenates the result
/// Assumes that the table has primary key consisting of one column named 'id'
template <typename TModel, typename TParser>
class TAsyncPagedReader: public TThrRefBase {
    using TModels = TVector<TModel>;
    using TThisRef = TIntrusivePtr<TAsyncPagedReader<TModel, TParser>>;

public:
    using TAsyncResult = NThreading::TFuture<TModels>;

    static TThisRef Create(NYdb::NTable::TTableClient& client, TString tablePath, TParser parser, TCallScopeCounter counter) {
        return new TAsyncPagedReader<TModel, TParser>(client, tablePath, parser, std::move(counter));
    }

    TAsyncResult Read(const NYdb::NTable::TReadTableSettings& settings) {
        Execute<NYdb::NTable::TAsyncTablePartIterator>(Client_, [=] (NYdb::NTable::TSession session) {
            return session.ReadTable(TablePath_, settings);
        }).Subscribe([self = TThisRef(this)] (const auto& f) {
            try {
                const NYdb::NTable::TTablePartIterator& v = f.GetValue();
                if (!v.IsSuccess()) {
                    self->Promise_.SetException(v.GetIssues().ToString());
                    return;
                }

                self->Iterator_ = v;
                self->Iterator_->ReadNext().Subscribe([self] (const auto& f) {
                    return self->AppendAndContinue(f);
                });
            } catch (...) {
                self->Promise_.SetException(std::current_exception());
            }
        });

        return Promise_.GetFuture();
    }

private:
    TAsyncPagedReader(NYdb::NTable::TTableClient& client, TString tablePath, TParser parser, TCallScopeCounter counters)
        : Client_{client}
        , TablePath_{std::move(tablePath)}
        , Parser_{parser}
        , Counters_{std::move(counters)}
    {
    }

    void AppendAndContinue(const NYdb::NTable::TAsyncSimpleStreamPart<NYdb::TResultSet>& f) {
        try {
            auto val = f.GetValue();
            if (!val.IsSuccess()) {
                if (val.EOS()) {
                    Promise_.SetValue(std::move(Result_));
                    return;
                }
                Promise_.SetException(val.GetIssues().ToString());
                return;
            }

            auto results = Parser_(val.GetPart());
            std::move(results.begin(), results.end(), std::back_inserter(Result_));

            Iterator_->ReadNext().Subscribe([self = TThisRef(this)] (const auto& f) {
                return self->AppendAndContinue(f);
            });
        } catch (...) {
            Counters_.RecordError();
            Promise_.SetException(std::current_exception());
        }
    }

private:
    std::optional<NYdb::NTable::TTablePartIterator> Iterator_;
    NYdb::NTable::TTableClient& Client_;
    TString TablePath_;
    TModels Result_;
    TParser Parser_;
    NThreading::TPromise<TModels> Promise_ = NThreading::NewPromise<TModels>();
    TCallScopeCounter Counters_;
};

template <typename TModel, typename TParser>
typename TAsyncPagedReader<TModel, TParser>::TAsyncResult ReadTable(
    NYdb::NTable::TTableClient& client, TString tablePath, TParser parser, TCallScopeCounter counters)
{
    NYdb::NTable::TReadTableSettings settings;
    return TAsyncPagedReader<TModel, TParser>::Create(client, tablePath, parser, std::move(counters))->Read(settings);
}

} // namespace NSolomon::NDb
