#pragma once

#include <ydb/public/sdk/cpp/client/ydb_result/result.h>
#include <ydb/public/sdk/cpp/client/ydb_table/table.h>

#include <util/generic/string.h>
#include <util/string/split.h>

#include <optional>

namespace NSolomon::NDb {
namespace NPrivate {

template <typename TInt>
auto CreateGetterForIntegralType(NYdb::TResultSetParser& parser, const TString& column) {
    return [column, &parser] () -> TMaybe<TInt> {
        if constexpr (std::is_same_v<TInt, i32>) {
            return parser.ColumnParser(column).GetOptionalInt32();
        } else if constexpr (std::is_same_v<TInt, i64>) {
            return parser.ColumnParser(column).GetOptionalInt64();
        } else if constexpr (std::is_same_v<TInt, ui32>) {
            return parser.ColumnParser(column).GetOptionalUint32();
        } else if constexpr (std::is_same_v<TInt, ui64>) {
            return parser.ColumnParser(column).GetOptionalUint64();
        } else if constexpr (std::is_same_v<TInt, bool>) {
            return parser.ColumnParser(column).GetOptionalBool();
        }
    };
}

} // namespace NPrivate

template <typename TModel>
class IFieldParser {
public:
    virtual ~IFieldParser() = default;
    virtual void LoadOpt(TModel& model) = 0;
};


template <typename TModel, typename TFieldType, typename TGetter>
class TFieldParserImpl: public IFieldParser<TModel> {
public:
    TFieldParserImpl(TFieldType field, TGetter&& getter)
        : Field_{field}
        , FieldGetter_{std::move(getter)}
    {
    }

    void LoadOpt(TModel& obj) override {
        auto value = FieldGetter_();
        if (value) {
            (obj.*Field_) = *value;
        }
    }

private:
    TFieldType Field_;
    TGetter FieldGetter_;
};

template <typename TModel, typename TSetter, typename TGetter>
class TIndirectFieldParserImpl: public IFieldParser<TModel> {
public:
    TIndirectFieldParserImpl(TGetter&& getter, TSetter&& setter)
        : Setter_{std::move(setter)}
        , Getter_{std::move(getter)}
    {
    }

    void LoadOpt(TModel& obj) override {
        try {
            auto value = Getter_();
            if (value) {
                Setter_(obj, *value);
            }
        } catch (...) {
            // TODO: use non actor based logger frontend
            if constexpr(requires(const TModel& m) { m.Id; }) {
                Cerr << "cannot load " << TypeName<TModel>() << " field from YDB, id=" << obj.Id
                     << ". reason: " << CurrentExceptionMessage() << Endl;
            } else {
                Cerr << "cannot load " << TypeName<TModel>() << " field from YDB"
                     << ". reason: " << CurrentExceptionMessage() << Endl;
            }
        }
    }

private:
    TSetter Setter_;
    TGetter Getter_;
};

template <typename TModel>
auto Utf8FieldParser(NYdb::TResultSetParser& parser, TString (TModel::*field), const TString& column) {
    auto getter = [=, &parser] {
        return parser.ColumnParser(column).GetOptionalUtf8();
    };

    return MakeHolder<TFieldParserImpl<TModel, decltype(field), decltype(getter)>>(field, std::move(getter));
}

template <typename TModel, typename TInt>
auto IntegralFieldParser(NYdb::TResultSetParser& parser, TInt (TModel::* field), TString column) {
    auto getter = NPrivate::CreateGetterForIntegralType<TInt>(parser, column);
    return MakeHolder<TFieldParserImpl<TModel, decltype(field), decltype(getter)>>(field, std::move(getter));
}
template <typename TModel>
auto BoolFieldParser(NYdb::TResultSetParser& parser, bool (TModel::*field), TString column) {
    return IntegralFieldParser<TModel, bool>(parser, field, column);
}

template <typename TModel>
auto Int64FieldParser(NYdb::TResultSetParser& parser, i64 (TModel::*field), TString column) {
    return IntegralFieldParser<TModel, i64>(parser, field, column);
}

template <typename TModel>
auto Int32FieldParser(NYdb::TResultSetParser& parser, i32 (TModel::* field), TString column) {
    return IntegralFieldParser<TModel, i32>(parser, field, column);
}

template <typename TModel>
auto Uint64FieldParser(NYdb::TResultSetParser& parser, ui64 (TModel::*field), TString column) {
    return IntegralFieldParser<TModel, ui64>(parser, field, column);
}

template <typename TModel>
auto Uint32FieldParser(NYdb::TResultSetParser& parser, ui32 (TModel::* field), TString column) {
    return IntegralFieldParser<TModel, ui32>(parser, field, column);
}

template <typename TModel>
auto DoubleFieldParser(NYdb::TResultSetParser& parser, double (TModel::* field), const TString& column) {
    auto getter = [=, &parser] {
        return parser.ColumnParser(column).GetOptionalDouble();
    };

    return MakeHolder<TFieldParserImpl<TModel, decltype(field), decltype(getter)>>(field, std::move(getter));
}

///// specialized parsers

template <typename TModel>
auto IToUI64FieldParser(NYdb::TResultSetParser& parser, ui64 (TModel::* field), const TString& column) {
    auto getter = [=, &parser] {
        return parser.ColumnParser(column).GetOptionalInt64();
    };

    auto setter = [=] (TModel& model, i64 value) {
        (model.*field) = static_cast<ui64>(value);
    };

    return MakeHolder<TIndirectFieldParserImpl<TModel, decltype(setter), decltype(getter)>>(std::move(getter), std::move(setter));
}

template <typename TModel>
auto IToUI32FieldParser(NYdb::TResultSetParser& parser, ui32 (TModel::* field), const TString& column) {
    auto getter = [=, &parser] {
        return parser.ColumnParser(column).GetOptionalInt32();
    };

    auto setter = [=] (TModel& model, i32 value) {
        (model.*field) = static_cast<ui32>(value);
    };

    return MakeHolder<TIndirectFieldParserImpl<TModel, decltype(setter), decltype(getter)>>(std::move(getter), std::move(setter));
}

template <typename TModel>
auto I32ToUI16FieldParser(NYdb::TResultSetParser& parser, ui16 (TModel::* field), const TString& column) {
    auto getter = [=, &parser] {
        return parser.ColumnParser(column).GetOptionalInt32();
    };

    auto setter = [=] (TModel& model, i32 value) {
        Y_ENSURE(value >= Min<ui16>() || value <= Max<ui16>(), "value " << value << " is out of bounds for ui16");
        (model.*field) = static_cast<ui16>(value);
    };

    return MakeHolder<TIndirectFieldParserImpl<TModel, decltype(setter), decltype(getter)>>(std::move(getter), std::move(setter));
}

template <typename TModel>
auto I64ToInstantFieldParser(
    NYdb::TResultSetParser& parser,
    TInstant (TModel::*field), const TString& column)
{
    auto getter = [=, &parser] {
        return parser.ColumnParser(column).GetOptionalInt64();
    };

    auto setter = [=] (TModel& model, i64 value) {
        (model.*field) = TInstant::MilliSeconds(value);
    };

    return MakeHolder<TIndirectFieldParserImpl<TModel, decltype(setter), decltype(getter)>>(std::move(getter), std::move(setter));
}

template <typename TModel, typename TInt>
auto IntegralToDurationFieldParser(NYdb::TResultSetParser& parser, TDuration (TModel::*field), const TString& column) {
    auto getter = NPrivate::CreateGetterForIntegralType<TInt>(parser, column);

    auto setter = [=] (TModel& model, TInt value) {
        (model.*field) = TDuration::Seconds(value);
    };

    return MakeHolder<TIndirectFieldParserImpl<TModel, decltype(setter), decltype(getter)>>(std::move(getter), std::move(setter));
}

template <typename TModel>
auto I32ToDurationFieldParser(NYdb::TResultSetParser& parser, TDuration (TModel::*field), TString column) {
    return IntegralToDurationFieldParser<TModel, i32>(parser, field, column);
}

template <typename TModel>
auto Uint32ToDurationFieldParser(NYdb::TResultSetParser& parser, TDuration (TModel::*field), TString column) {
    return IntegralToDurationFieldParser<TModel, ui32>(parser, field, column);
}

template <typename TModel>
auto Utf8ToVectorParser(
    NYdb::TResultSetParser& parser,
    TVector<TString> (TModel::*field), const TString& column)
{
    auto getter = [=, &parser] {
        return parser.ColumnParser(column).GetOptionalUtf8();
    };

    auto setter = [=] (TModel& model, const TString& value) {
        StringSplitter(value).Split('\t').Collect(&(model.*field));
    };

    return MakeHolder<TIndirectFieldParserImpl<TModel, decltype(setter), decltype(getter)>>(std::move(getter), std::move(setter));
}

template <typename TModel, typename TEnum>
auto Utf8ToEnumFieldParser(
    NYdb::TResultSetParser& parser,
    TEnum (TModel::*field), const TString& column)
{
    auto getter = [=, &parser] {
        return parser.ColumnParser(column).GetOptionalUtf8();
    };

    auto setter = [=] (TModel& model, const TString& value) {
        (model.*field) = ::FromString<TEnum>(value);
    };

    return MakeHolder<TIndirectFieldParserImpl<TModel, decltype(setter), decltype(getter)>>(std::move(getter), std::move(setter));
}

template <typename TModel>
auto Utf8ToUint32FieldParser(
    NYdb::TResultSetParser& parser,
    ui32 (TModel::*field), const TString& column)
{
    auto getter = [=, &parser] {
        return parser.ColumnParser(column).GetOptionalUtf8();
    };

    auto setter = [=] (TModel& model, const TString& value) {
        (model.*field) = value.empty() ? 0 : ::FromString<ui32>(value);
    };

    return MakeHolder<TIndirectFieldParserImpl<TModel, decltype(setter), decltype(getter)>>(std::move(getter), std::move(setter));
}

template <typename T>
class TModelParser {
    using TMapping = THashMap<TString, THolder<IFieldParser<T>>>;

public:
    explicit TModelParser(const NYdb::TResultSet& rs)
        : Parser_{rs}
        , Meta_{rs.GetColumnsMeta()}
    {
    }

    TVector<T> Parse() {
        TVector<T> result;

        while (Parser_.TryNextRow()) try {
            auto& model = result.emplace_back();
            for (auto&& column: Meta_) {
                auto it = FieldMappings_.find(column.Name);
                if (it != FieldMappings_.end()) {
                    it->second->LoadOpt(model);
                }
            }
        } catch (...) {
            Cerr << "Error while parsing query result: " << CurrentExceptionMessage() << Endl;
        }

        return result;
    }

protected:
    NYdb::TResultSetParser Parser_;
    const TVector<NYdb::TColumn>& Meta_;
    TMapping FieldMappings_;
};

template <typename TModel, typename TParser>
TVector<TModel> ParseSelectResult(const NYdb::NTable::TDataQueryResult& result) {
    auto&& resultSets = result.GetResultSets();

    Y_ENSURE(resultSets.size() <= 1);
    if (resultSets.size() == 0) {
        return {};
    }

    return TParser{resultSets[0]}.Parse();
}

template <typename TModel, typename TParser>
std::optional<TModel> ParseOne(const NYdb::NTable::TDataQueryResult& result) {
    auto&& resultSets = result.GetResultSets();
    if (resultSets.size() < 1 || resultSets[0].RowsCount() < 1) {
        return {};
    }

    TParser p{resultSets[0]};
    auto models = p.Parse();
    return models.empty() ? std::nullopt : std::optional<TModel>{models[0]};
}

template <typename TModel, typename TParser>
TVector<TModel> ParseReadTableResult(const NYdb::TResultSet& resultSet) {
    return TParser{resultSet}.Parse();
}

} // namespace NSolomon::NDb
