#pragma once

#include <ydb/public/sdk/cpp/client/ydb_params/params.h>
#include <ydb/public/sdk/cpp/client/ydb_table/table.h>

#include <util/string/builder.h>

#include <utility>

namespace NSolomon {
namespace NDb {
    /// TODO(msherbakov): probably it's better to not to use exceptions at all here

    inline NYdb::NTable::TRetryOperationSettings RetrySettings() {
        return NYdb::NTable::TRetryOperationSettings{}
            .SlowBackoffSettings(
                NYdb::NTable::TRetryOperationSettings::DefaultSlowBackoffSettings()
                    .SlotDuration(TDuration::MilliSeconds(500))
            );
    }

    template <typename TResult>
    TResult Execute(
        NYdb::NTable::TTableClient& client, std::function<TResult(NYdb::NTable::TSession session)> op)
    {
        return client.GetSession().Apply([=] (const NYdb::NTable::TAsyncCreateSessionResult& f) {
            auto&& r = f.GetValueSync();

            if (!r.IsSuccess()) {
                ythrow yexception() << "Cannot create session: " << r.GetIssues().ToString();
            }

            auto&& session = r.GetSession();

            return op(session);
        });
    }

    static NYdb::NTable::TTxControl DefaultTx() {
        return NYdb::NTable::TTxControl::BeginTx().CommitTx();
    }

    inline NYdb::NTable::TAsyncDataQueryResult ExecutePrepared(
        NYdb::NTable::TSession session, const TString& query, NYdb::TParams params)
    {
        return session.PrepareDataQuery(query).Apply([=] (const NYdb::NTable::TAsyncPrepareQueryResult& f) mutable {
            auto&& r = f.GetValueSync();

            if (!r.IsSuccess()) {
                ythrow yexception() << "Error while preparing request: " << r.GetIssues().ToString();
            }

            auto q = r.GetQuery();
            return q.Execute(DefaultTx(), std::move(params));
        });
    }

    inline NYdb::NTable::TAsyncDataQueryResult ExecutePrepared(
        NYdb::NTable::TTableClient client, const TString& query, const NYdb::TParams& params)
    {
        return client.GetSession().Apply([=] (const NYdb::NTable::TAsyncCreateSessionResult& f) {
            auto&& r = f.GetValueSync();

            if (!r.IsSuccess()) {
                ythrow yexception() << "Cannot create session: " << r.GetIssues().ToString();
            }

            auto&& session = r.GetSession();
            return ExecutePrepared(session, query, params);
        });
    }

    inline NYdb::NTable::TAsyncDataQueryResult ExecutePreparedWithRetries(
        NYdb::NTable::TTableClient client,
        const TString& query,
        NYdb::TParams params,
        NYdb::NTable::TRetryOperationSettings retrySettings)
    {
        auto p = NThreading::NewPromise<NYdb::NTable::TDataQueryResult>();

        auto cb = [p] (auto f) mutable {
            auto&& v = f.ExtractValue();

            if (v.IsSuccess()) {
                p.SetValue(v);
            }

            return v;
        };

        client.RetryOperation<NYdb::NTable::TDataQueryResult>([=, param{std::move(params)}] (NYdb::NTable::TSession s) mutable {
            return ExecutePrepared(std::move(s), query, param).Apply(cb);
        }, retrySettings).Subscribe([=] (auto f) mutable {
            if (p.HasValue() || p.HasException()) {
                return;
            }

            try {
                auto&& v = f.GetValueSync();
                Y_UNUSED(v);
            } catch (...) {
                p.SetException(std::current_exception());
                return;
            }

            p.SetException(f.GetValue().GetIssues().ToString());
        });

        return p;
    }

    template <typename T>
    inline NThreading::TFuture<void> CheckStatus(TStringBuf message, const NThreading::TFuture<T>& future) {
        auto promise = NThreading::NewPromise<void>();
        future.Subscribe([promise, message](const NThreading::TFuture<T>& result) mutable {
            try {
                if (auto& v = result.GetValueSync(); v.IsSuccess()) {
                    promise.SetValue();
                } else {
                    promise.SetException(TStringBuilder{} << message << ", issues: " << v.GetIssues().ToString());
                }
            } catch (...) {
                promise.SetException(std::current_exception());
            }
        });
        return promise.GetFuture();
    }

    template <typename T, typename TParseFunc>
    void ReadDataWithRetries(NYdb::NTable::TTableClient& client, const TString& query, NYdb::TParams params,
        NThreading::TPromise<T> p, TParseFunc parser)
    {
        auto handleResult = [=] (const NYdb::NTable::TAsyncDataQueryResult& f) mutable {
            if (f.HasException()) {
                try {
                    f.GetValueSync();
                } catch (...) {
                    p.SetException(std::current_exception());
                }
            }

            auto&& v = f.GetValueSync();

            if (v.IsSuccess()) {
                p.SetValue(parser(v));
            }

            return NThreading::MakeFuture(NYdb::TStatus(v));
        };

        client.RetryOperation<NYdb::TStatus>([=] (NYdb::NTable::TSession s) mutable {
            return ExecutePrepared(std::move(s), query, std::move(params)).Apply(handleResult);
        }).Subscribe([p] (auto f) mutable {
            try {
                auto&& v = f.GetValue();

                if (!v.IsSuccess()) {
                    p.SetException(v.GetIssues().ToString());
                }
            } catch (...) {
                p.SetException(std::current_exception());
            }
        });
    }
} // namespace NDb
} // namespace NSolomon
