package cache

import (
	"container/list"
	"encoding/json"
	"fmt"
	"log"
	"math/rand"
	"reflect"
	"strings"
	"sync"
	"time"
)

const (
	noUpdate = iota
	wantUpdate
	forceUpdate
)

// ==========================================================================================

type Unmarshaller struct {
	Unmarshal func(raw []byte) (interface{}, error)
}

func NewUnmarshaller(in interface{}) *Unmarshaller {
	u := &Unmarshaller{}
	inType := reflect.TypeOf(in)
	if inType.Kind() == reflect.Ptr || inType.Kind() == reflect.Interface {
		u.Unmarshal = func(raw []byte) (interface{}, error) {
			ptr := reflect.New(inType.Elem()).Interface()
			err := json.Unmarshal(raw, ptr)
			return ptr, err
		}
	} else {
		u.Unmarshal = func(raw []byte) (interface{}, error) {
			ptr := reflect.New(inType)
			err := json.Unmarshal(raw, ptr.Interface())
			return reflect.Indirect(ptr).Interface(), err
		}
	}
	return u
}

// ==========================================================================================

type CacheError struct {
	DataErr error
	LastErr error
	EOL     time.Time
}

// Get data error
func (e *CacheError) Unwrap() error {
	if e == nil {
		return nil
	}
	return e.DataErr
}

// Get last error
func (e *CacheError) UnwrapLast() error {
	if e == nil {
		return nil
	}
	return e.LastErr
}

func (e *CacheError) Error() string {
	if e == nil {
		return "<nil>"
	}
	var s string
	if expired := e.Expired(); expired > 0 {
		s = "expired " + expired.String() + " ago"
	}
	if e.DataErr != nil {
		if s != "" {
			s += ", "
		}
		s += "data error - " + strings.TrimRight(e.DataErr.Error(), "\n\r ")
	}
	if e.LastErr != nil {
		if s != "" {
			s += ", "
		}
		s += "last error - " + strings.TrimRight(e.LastErr.Error(), "\n\r ")
	}
	if s == "" {
		return "<nil>"
	}
	return s
}

// Data is stale
func (e *CacheError) IsStale() bool {
	if e == nil {
		return false
	}
	return e.Expired() > 0
}

// Data expired this many time ago
func (e *CacheError) Expired() time.Duration {
	if e == nil || e.EOL.IsZero() {
		return time.Duration(0)
	}
	return time.Since(e.EOL)
}

// ==========================================================================================

// Threadsafe 2Q-like Cache that is transparently Getting expired or absent values.
// If Getting a value is delayed, then several Get calls will wait, not overloading
// the source of values.
// Able to serve stale items and prefetch.
//
// type Cache interface {
//     Len()
//     Keys() []interface{}
//     Contains(key interface{}) bool
//     Peek(key interface{}) (interface{}, error)
//     Get(key interface{}) (interface{}, error)
//     Set(key interface{}, val interface{}) (interface{}, error)
//     Delete(key interface{}) (interface{}, error)
//     Purge()
//	   Destroy()
//	   Dump() ([]byte, error)
//     Restore([]byte, bumpEOL bool, keyInt, valueInt interface{}) error
// }
//

type Getter func(interface{}) (interface{}, error)

type Cache struct {
	LogPrefix          string
	Getter             Getter
	GoodCacheTime      time.Duration
	BadCacheTime       time.Duration
	PrefetchTime       time.Duration
	prefetchJitterFrac float64
	CleanUpInterval    time.Duration
	VerboseLevel       int
	ServeStale         bool
	MaxSize            int
	max1Size           int
	items              map[interface{}]*cacheItem
	listEOLGood        *list.List
	listEOLBad         *list.List
	listLRU1           *list.List
	listLRU2           *list.List
	mutex              sync.Mutex
	stopChan           chan struct{}
}

type cacheItem struct {
	key        interface{}
	val        interface{}
	eol        time.Time
	lastUpdate time.Time
	err        error
	isUpdating bool
	eEOLGood   *list.Element
	eEOLBad    *list.Element
	eLRU1      *list.Element
	eLRU2      *list.Element
	waitChan   chan struct{}
}

type DumpItem struct {
	Key interface{} `json:"k"`
	Val interface{} `json:"v"`
	Eol int64       `json:"t"`
}

type RestoreItem struct {
	Key json.RawMessage `json:"k"`
	Val json.RawMessage `json:"v"`
	Eol int64           `json:"t"`
}

// If serveStale == true, set maxSize > 0, or cache will grow indefinitely
// Rule of thumb: cleanUpInterval << prefetchTime <= badCacheTime < goodCacheTime, for example 5s << 30s <= 1m < 10m
//
func NewCache(name string,
	getFunc Getter,
	goodCacheTime, badCacheTime, prefetchTime, cleanUpInterval time.Duration,
	serveStale bool,
	verboseLevel int,
	maxSize int) *Cache {

	qqRatio := 0.5
	c := &Cache{
		LogPrefix:          fmt.Sprintf("[%s cache] ", name),
		Getter:             getFunc,
		GoodCacheTime:      goodCacheTime,
		BadCacheTime:       badCacheTime,
		PrefetchTime:       prefetchTime,
		prefetchJitterFrac: 0.5,
		CleanUpInterval:    cleanUpInterval,
		ServeStale:         serveStale,
		VerboseLevel:       verboseLevel,
		MaxSize:            maxSize,
		max1Size:           int(float64(maxSize) * qqRatio),
		items:              make(map[interface{}]*cacheItem),
		listEOLGood:        list.New(),
		listEOLBad:         list.New(),
		listLRU2:           list.New(),
		listLRU1:           list.New(),
		stopChan:           make(chan struct{}, 1),
	}
	go c.cleanupRoutine()
	return c
}

// ==========================================================================================

func (c *Cache) log(lvl int, ts *time.Time, format string, v ...interface{}) {
	if c.VerboseLevel >= lvl {
		tsStr := ""
		if ts != nil {
			tsStr = ", " + time.Since(*ts).String()
		}
		log.Printf(c.LogPrefix+format+tsStr, v...)
	}
}

func (c *Cache) cleanupRoutine() {
	var wg sync.WaitGroup

	ticker := time.NewTicker(c.CleanUpInterval)
	defer ticker.Stop()
	for {
		select {
		case <-ticker.C:
			prefetchingItems := 0
			prefetchedItems := 0
			removedItems := 0
			var prefTime time.Time
			var prefetchJitterNs int64

			c.mutex.Lock()
			nowTime := time.Now()
			if int64(c.PrefetchTime) != 0 {
				prefetchJitterNs = int64(c.prefetchJitterFrac * float64(c.PrefetchTime.Nanoseconds()))
				prefTime = nowTime.Add(c.PrefetchTime + c.CleanUpInterval)
			}
			for _, l := range []*list.List{c.listEOLGood, c.listEOLBad} {
				for e := l.Back(); e != nil; e = e.Prev() {
					item := e.Value.(*cacheItem)
					if item.isUpdating {
						continue
					}
					if item.eol.Before(prefTime) {
						prefetchingItems++
						wg.Add(1)
						go func(item *cacheItem) {
							defer wg.Done()
							time.Sleep(time.Duration(rand.Int63n(prefetchJitterNs)) * time.Nanosecond)

							c.mutex.Lock()
							// Item has gone or is updating
							cItem, ok := c.items[item.key]
							if !ok || cItem != item || item.isUpdating {
								c.mutex.Unlock()
								return
							} else {
								item.isUpdating = true
								prefetchedItems++
							}
							c.mutex.Unlock()

							_, _ = c.updateSafe(item)
						}(item)
					} else if !c.ServeStale && item.eol.Before(nowTime) {
						removedItems++
						c.removeItem(item)
					} else {
						break
					}
				}
			}
			cacheLength := len(c.items)
			c.mutex.Unlock()

			lvl := 2
			if prefetchingItems != 0 || removedItems != 0 {
				lvl = 1
			}
			c.log(lvl, &nowTime, "cleanup scheduled, records total=%d removed=%d prefetching=%d",
				cacheLength,
				removedItems,
				prefetchingItems,
			)
			wg.Wait()
			c.mutex.Lock()
			cacheLength = len(c.items)
			c.mutex.Unlock()
			c.log(lvl, &nowTime, "cleanup done, records total=%d prefetched=%d",
				cacheLength,
				prefetchedItems,
			)
		case <-c.stopChan:
			c.log(1, nil, "stopping cleanup routine")
			return
		}
	}
}

func (c *Cache) registerAccess(item *cacheItem) {
	if c.MaxSize == 0 {
		return
	}
	if item.eLRU2 != nil {
		c.listLRU2.MoveToFront(item.eLRU2)
	} else if item.eLRU1 != nil {
		item.eLRU1, item.eLRU2 = nil, c.listLRU2.PushFront(c.listLRU1.Remove(item.eLRU1))
	} else {
		item.eLRU1 = c.listLRU1.PushFront(item)
	}
	c.checkSpace()
}

func (c *Cache) checkSpace() {
	var e *list.Element

	len1 := c.listLRU1.Len()
	len2 := c.listLRU2.Len()
	if len1+len2 <= c.MaxSize {
		return
	}
	if len1 >= c.max1Size {
		e = c.listLRU1.Back()
	} else {
		e = c.listLRU2.Back()
	}
	if e != nil {
		c.removeItem(e.Value.(*cacheItem))
	}
}

func (c *Cache) setEOLGoodTime(item *cacheItem) {
	item.eol = time.Now().Add(c.GoodCacheTime)
	if item.eEOLGood != nil {
		c.listEOLGood.MoveToFront(item.eEOLGood)
	} else if item.eEOLBad != nil {
		item.eEOLBad, item.eEOLGood = nil, c.listEOLGood.PushFront(c.listEOLBad.Remove(item.eEOLBad))
	} else {
		item.eEOLGood = c.listEOLGood.PushFront(item)
	}
}

func (c *Cache) setEOLBadTime(item *cacheItem) {
	item.eol = time.Now().Add(c.BadCacheTime)
	if item.eEOLBad != nil {
		c.listEOLBad.MoveToFront(item.eEOLBad)
	} else if item.eEOLGood != nil {
		item.eEOLGood, item.eEOLBad = nil, c.listEOLBad.PushFront(c.listEOLGood.Remove(item.eEOLGood))
	} else {
		item.eEOLBad = c.listEOLBad.PushFront(item)
	}
}

func (c *Cache) removeItem(item *cacheItem) {
	delete(c.items, item.key)
	if item.eLRU1 != nil {
		_ = c.listLRU1.Remove(item.eLRU1)
	}
	if item.eLRU2 != nil {
		_ = c.listLRU2.Remove(item.eLRU2)
	}
	if item.eEOLGood != nil {
		_ = c.listEOLGood.Remove(item.eEOLGood)
	}
	if item.eEOLBad != nil {
		_ = c.listEOLBad.Remove(item.eEOLBad)
	}
}

func (c *Cache) updateSafe(item *cacheItem) (interface{}, error) {
	reqTime := time.Now()
	val, err := c.Getter(item.key)
	c.log(2, &reqTime, "got value for %v from getter (err=%v)", item.key, err)

	c.mutex.Lock()
	defer c.mutex.Unlock()

	if err != nil {
		// When serving stale record, we need to update EOL time, so that we will not overload
		// value source. Value EOL we will keep in CacheError
		// In case item.err != nil, create item.err anew: maybe somebody is already using it
		if item.err == nil {
			item.err = &CacheError{DataErr: nil, LastErr: err, EOL: item.eol}
		} else {
			cerr := item.err.(*CacheError)
			item.err = &CacheError{DataErr: cerr.Unwrap(), LastErr: err, EOL: cerr.EOL}
		}
		if c.ServeStale && item.val != nil {
			c.log(2, nil, "serving stale value for %v (err=%v)", item.key, item.err)
		} else {
			item.val = val
			item.err.(*CacheError).DataErr = err
		}
		c.setEOLBadTime(item)
	} else {
		item.val, item.err = val, nil
		c.setEOLGoodTime(item)
	}
	item.lastUpdate = reqTime
	item.isUpdating = false
	if item.waitChan != nil {
		close(item.waitChan)
		item.waitChan = nil
	}
	return item.val, item.err
}

func (c *Cache) getSafe(key interface{}, update int, minLastUpdate *time.Time) (interface{}, error) {
	var val interface{}
	var err error
	var ok bool
	var isFresh bool
	var item *cacheItem
	var waitChan chan struct{}

	reqTime := time.Now()
	c.mutex.Lock()
	if item, ok = c.items[key]; ok {
		if update != noUpdate {
			c.registerAccess(item)
		}
		if update != forceUpdate {
			if minLastUpdate != nil {
				isFresh = item.lastUpdate.After(*minLastUpdate)
			} else {
				isFresh = item.eol.After(reqTime)
			}
			// XXX if we Peek expired value with ServeStale == true, do we need to answer?
			// The problem is that Get after that Peek will try Getting fresh value
			if isFresh || (c.ServeStale && update == noUpdate) {
				if !isFresh && item.err == nil {
					item.err = &CacheError{EOL: item.eol}
				}
				val, err = item.val, item.err
				c.mutex.Unlock()
				c.log(2, &reqTime, "got value for %v (err=%v)", key, err)
				return val, err
			}
		}
		// waitChan != nil => there is at least one waiter for the value
		if item.isUpdating {
			if item.waitChan == nil {
				item.waitChan = make(chan struct{})
			}
			waitChan = item.waitChan
		} else if update != noUpdate {
			item.isUpdating = true
		}
	} else if update != noUpdate {
		item = &cacheItem{
			key:        key,
			isUpdating: true,
		}
		c.items[key] = item
		c.registerAccess(item)
	}
	c.mutex.Unlock()

	if waitChan != nil {
		c.log(2, nil, "waiting in queue for %v", key)
		<-waitChan
		c.log(2, &reqTime, "waiting succeded for %v", key)
		c.mutex.Lock()
		val, err = item.val, item.err
		c.mutex.Unlock()
		c.log(2, &reqTime, "got value for %v (err=%v)", key, err)
	} else if update != noUpdate {
		val, err = c.updateSafe(item)
		c.log(2, &reqTime, "got value for %v (err=%v)", key, err)
	} else {
		c.log(2, &reqTime, "got no value for %v", key)
	}
	return val, err
}

func (c *Cache) purge() {
	for key := range c.items {
		delete(c.items, key)
	}
	c.listEOLGood.Init()
	c.listEOLBad.Init()
	c.listLRU1.Init()
	c.listLRU2.Init()
}

// ==========================================================================================

func (c *Cache) Len() int {
	c.mutex.Lock()
	defer c.mutex.Unlock()

	return len(c.items)
}

func (c *Cache) Keys() []interface{} {
	c.mutex.Lock()
	defer c.mutex.Unlock()

	keys := make([]interface{}, len(c.items))
	i := 0
	for _, l := range []*list.List{c.listLRU1, c.listLRU2} {
		for e := l.Back(); e != nil; e = e.Prev() {
			keys[i] = e.Value.(*cacheItem).key
			i++
		}
	}
	return keys
}

func (c *Cache) Contains(key interface{}) bool {
	c.mutex.Lock()
	defer c.mutex.Unlock()

	_, ok := c.items[key]
	return ok
}

func (c *Cache) Peek(key interface{}) (interface{}, error) {
	return c.getSafe(key, noUpdate, nil)
}

func (c *Cache) Get(key interface{}) (interface{}, error) {
	return c.getSafe(key, wantUpdate, nil)
}

func (c *Cache) GetForceUpdate(key interface{}) (interface{}, error) {
	return c.getSafe(key, forceUpdate, nil)
}

func (c *Cache) GetForceFresh(key interface{}, minLastUpdate *time.Time) (interface{}, error) {
	return c.getSafe(key, wantUpdate, minLastUpdate)
}

func (c *Cache) Set(key interface{}, val interface{}) (interface{}, error) {
	var err error
	var ok bool
	var item *cacheItem

	c.log(2, nil, "setting value for %v", key)

	c.mutex.Lock()
	defer c.mutex.Unlock()

	if item, ok = c.items[key]; ok {
		val, item.val = item.val, val
		err, item.err = item.err, nil
	} else {
		item = &cacheItem{
			key: key,
			val: val,
		}
		c.items[key] = item
		val, err = nil, nil
	}
	c.setEOLGoodTime(item)
	c.registerAccess(item)
	return val, err
}

func (c *Cache) Delete(key interface{}) (interface{}, error) {
	var val interface{}
	var err error

	c.log(2, nil, "trying to remove value for %v", key)

	c.mutex.Lock()
	defer c.mutex.Unlock()

	if item, ok := c.items[key]; ok {
		val, err = item.val, item.err
		c.removeItem(item)
	}
	return val, err
}

func (c *Cache) Purge() {
	c.log(1, nil, "purging")

	c.mutex.Lock()
	defer c.mutex.Unlock()

	c.purge()
}

func (c *Cache) Destroy() {
	c.log(1, nil, "destroying")
	c.stopChan <- struct{}{}

	c.mutex.Lock()
	defer c.mutex.Unlock()

	c.purge()
}

func (c *Cache) Dump(onlyGood bool) ([]byte, error) {
	reqTime := time.Now()
	c.log(1, nil, "dumping")

	c.mutex.Lock()
	data := make([]DumpItem, 0, len(c.items))
	for _, l := range []*list.List{c.listLRU1, c.listLRU2} {
		for e := l.Back(); e != nil; e = e.Prev() {
			item := e.Value.(*cacheItem)
			if item.isUpdating || (onlyGood && item.eEOLGood == nil) {
				continue
			}
			data = append(data, DumpItem{
				Key: &item.key,
				Val: &item.val,
				Eol: item.eol.Unix(),
			})
		}
	}
	c.mutex.Unlock()

	r, err := json.Marshal(data)
	c.log(1, &reqTime, "dumped")
	return r, err
}

func (c *Cache) Restore(inData []byte, bumpEOL bool, keyInt, valInt interface{}) error {
	reqTime := time.Now()
	var data []RestoreItem

	c.log(1, nil, "restoring")
	keyUnmarshaller := NewUnmarshaller(keyInt)
	valUnmarshaller := NewUnmarshaller(valInt)
	if err := json.Unmarshal(inData, &data); err != nil {
		return err
	}

	c.mutex.Lock()
	defer c.mutex.Unlock()

	c.purge()
	itemTime := reqTime.Add(c.GoodCacheTime)
	items := map[interface{}]*cacheItem{}
	for _, restoreItem := range data {
		key, err := keyUnmarshaller.Unmarshal(restoreItem.Key)
		if err != nil {
			return err
		}
		if _, ok := items[key]; ok {
			continue
		}
		val, err := valUnmarshaller.Unmarshal(restoreItem.Val)
		if err != nil {
			return err
		}
		if !bumpEOL {
			itemTime = time.Unix(restoreItem.Eol, 0)
		}
		item := &cacheItem{
			key: key,
			val: val,
			eol: itemTime,
		}
		item.eEOLGood = c.listEOLGood.PushFront(item)
		c.registerAccess(item)
		items[key] = item
	}
	c.items = items
	c.log(1, &reqTime, "restored")
	return nil
}
