package filelogger

import (
	"compress/gzip"
	"fmt"
	"io"
	"log"
	"os"
	"strconv"
	"sync"
)

// =================================================================================================

// Examples:
//
func SetStdLogger() {
	log.SetFlags(log.Ldate | log.Ltime | log.Lmicroseconds)
}

func SetFileLogger(filePath string, maxSize int64, count uint) {
	log.SetFlags(log.Ldate | log.Ltime | log.Lmicroseconds)
	log.SetOutput(NewFileLogger(filePath, maxSize, count))
}

// =================================================================================================

type FileLogger struct {
	m        sync.Mutex
	rotating bool
	path     string
	mode     os.FileMode
	size     int64
	maxSize  int64
	count    uint
	file     *os.File
	gzLevel  int
}

func NewFileLogger(path string, maxSize int64, count uint) *FileLogger {
	f := &FileLogger{
		path:    path,
		mode:    0644,
		maxSize: maxSize,
		count:   count,
		gzLevel: 3,
	}
	if err := f.openfile(); err != nil {
		_, _ = fmt.Fprintln(os.Stdout, "cannot log to file, switching to stdout, "+err.Error())
	}
	return f
}

func (f *FileLogger) Write(data []byte) (size int, err error) {
	f.m.Lock()
	defer f.m.Unlock()

	if f.file != nil && f.size >= f.maxSize && !f.rotating {
		if err = os.Rename(f.path, f.path+".1"); err != nil {
			f.stop(err)
		} else {
			oldFile := f.file
			if err = f.openfile(); err != nil {
				f.file = oldFile
				f.stop(err)
			} else {
				f.rotating = true
				go f.rotate(oldFile)
			}
		}
	}
	if f.file != nil {
		if size, err = f.file.Write(data); err != nil {
			f.stop(err)
		} else {
			f.size += int64(size)
		}
	} else {
		size, err = os.Stdout.Write(data)
	}
	return
}

// Called under lock or at init
func (f *FileLogger) openfile() error {
	f.file, f.size = nil, 0

	file, err := os.OpenFile(f.path, os.O_APPEND|os.O_CREATE|os.O_WRONLY, f.mode)
	if err != nil {
		return err
	}
	stat, err := file.Stat()
	if err != nil {
		go func() {
			_ = file.Close()
		}()
		return err
	}
	f.file, f.size = file, stat.Size()

	return nil
}

// Called without lock
func (f *FileLogger) rotate(oldFile *os.File) {
	var err error
	defer func() {
		f.m.Lock()
		if err != nil && f.file != nil {
			f.stop(err)
		}
		f.rotating = false
		f.m.Unlock()
	}()

	if err = oldFile.Sync(); err != nil {
		return
	} else if err = oldFile.Close(); err != nil {
		return
	}
	lastName := f.path + ".1"
	if err = f.compress(lastName, lastName+".gz-tmp"); err != nil {
		return
	}
	if err = os.Remove(lastName); err != nil {
		return
	}
	for idx := f.count - 1; idx > 0; idx-- {
		newName := f.path + "." + strconv.FormatInt(int64(idx), 10) + ".gz"
		oldName := f.path + "." + strconv.FormatInt(int64(idx+1), 10) + ".gz"
		if _, err = os.Stat(newName); err != nil {
			if !os.IsNotExist(err) {
				return
			}
		} else if err = os.Rename(newName, oldName); err != nil {
			return
		}
	}
	err = os.Rename(lastName+".gz-tmp", lastName+".gz")
}

func (f *FileLogger) compress(srcName, dstName string) error {
	srcFile, err := os.OpenFile(srcName, os.O_RDONLY, 0)
	if err != nil {
		return err
	}
	defer func() {
		_ = srcFile.Close()
	}()

	dstFile, err := os.OpenFile(dstName, os.O_TRUNC|os.O_CREATE|os.O_WRONLY, f.mode)
	if err != nil {
		return err
	}
	defer func() {
		_ = dstFile.Close()
	}()

	zFile, err := gzip.NewWriterLevel(dstFile, f.gzLevel)
	if err != nil {
		return err
	}
	defer func() {
		_ = zFile.Close()
	}()

	_, err = io.Copy(zFile, srcFile)
	return err
}

// Called under lock
func (f *FileLogger) stop(err error) {
	file := f.file
	f.file = nil
	go func() {
		_ = file.Sync()
		_ = file.Close()
	}()
	_, _ = fmt.Fprintln(os.Stdout, "cannot log to file, switching to stdout, "+err.Error())
}
