package iam

import (
	iamApi "a.yandex-team.ru/cloud/bitbucket/private-api/yandex/cloud/priv/iam/v1"
	"a.yandex-team.ru/solomon/libs/go/grpc"
	"a.yandex-team.ru/solomon/libs/go/iam/internal/cloud"
	"a.yandex-team.ru/solomon/protos/secrets"
	"context"
	"crypto/rsa"
	"encoding/json"
	"fmt"
	"github.com/golang-jwt/jwt/v4"
	"os"
	"time"
)

type Key struct {
	ID               string `json:"id"`
	ServiceAccountID string `json:"service_account_id"`
	PublicKey        string `json:"public_key"`
	PrivateKey       string `json:"private_key"`
}

var jwtMethod = &jwt.SigningMethodRSAPSS{
	SigningMethodRSA: jwt.SigningMethodPS256.SigningMethodRSA,
	Options: &rsa.PSSOptions{
		SaltLength: rsa.PSSSaltLengthEqualsHash,
	},
}

var tsAddress = map[secrets.CloudEnv]string{
	secrets.CloudEnv_PREPROD: "ts.private-api.cloud-preprod.yandex.net:4282",
	secrets.CloudEnv_PROD:    "ts.private-api.cloud.yandex.net:4282",
	secrets.CloudEnv_GPN:     "ts.private-api.ycp.gpn.yandexcloud.net:443",
}

func makeSignedToken(keyJSON string) (string, error) {
	var key Key
	if err := json.Unmarshal([]byte(keyJSON), &key); err != nil {
		return "", err
	}

	privateKey, err := jwt.ParseRSAPrivateKeyFromPEM([]byte(key.PrivateKey))
	if err != nil {
		return "", err
	}

	issuedAt := time.Now()
	token := jwt.NewWithClaims(jwtMethod, jwt.StandardClaims{
		Issuer:    key.ServiceAccountID,
		Audience:  "https://iam.api.cloud.yandex.net/iam/v1/tokens",
		IssuedAt:  issuedAt.Unix(),
		ExpiresAt: issuedAt.Add(10 * time.Minute).Unix(),
	})
	token.Header["kid"] = key.ID

	return token.SignedString(privateKey)
}

func GetIamTokenFromTokenService(env secrets.CloudEnv, keyJSON string) (string, error) {
	ctx, cancel := context.WithTimeout(context.Background(), 20*time.Second)
	defer cancel()

	signedToken, err := makeSignedToken(keyJSON)
	if err != nil {
		return "", fmt.Errorf("cannot create signed JWT token, %v", err)
	}

	addr := tsAddress[env]
	conn, err := grpc.NewConnection(ctx, addr)
	if err != nil {
		return "", fmt.Errorf("cannot connect to %s, %v", addr, err)
	}
	defer conn.Close()

	request := &iamApi.CreateIamTokenRequest{
		Identity: &iamApi.CreateIamTokenRequest_Jwt{Jwt: signedToken},
	}

	client := iamApi.NewIamTokenServiceClient(conn)
	response, err := client.Create(ctx, request)
	if err != nil {
		return "", fmt.Errorf("cannot create IAM token by JWT, %v", err)
	}

	return response.IamToken, nil
}

func GetIamTokenFromLocalMetaDataService() (string, error) {
	mds := cloud.MetaDataService{}
	if mds.IsAvailable() {
		fmt.Fprintln(os.Stderr, "getting IAM token from local meta-data service")
		return mds.GetIamToken()
	}

	return "", nil
}

func GetIamToken(env secrets.CloudEnv, keyJSON string) (string, error) {
	mds := cloud.MetaDataService{}
	if mds.IsAvailable() {
		fmt.Fprintln(os.Stderr, "getting IAM token from local meta-data service")
		return mds.GetIamToken()
	}

	fmt.Fprintln(os.Stderr, "getting IAM token from IamTokenService")
	return GetIamTokenFromTokenService(env, keyJSON)
}
