package localactor

import (
	"fmt"
	"log"
	"sync/atomic"
	"time"
)

// ==========================================================================================

// LocalActor acts as a wrapper for an external object, consuming looper interface
//
// type Looper interface {
//     Prepare() error
//     Loop(interface{}) (interface{}, error)
//     Shutdown()
// }
//
// type LocalActor interface {
//     Start() error
//     Stop() error
//     StopAsync() error // stop from the loop or if no delay needed
//     AddTicker(string, time.Duration, interface{}) error
//     DeleteTicker(string) error
//     Send(interface{}, waitResult bool) (interface{}, error)
// }
//
const (
	ActorStopped int32 = iota
	ActorStarting
	ActorStarted
	ActorStopping
)

type ControlMsg struct {
	Msg        interface{}
	Error      error
	ResultChan chan *ControlMsg
}

type Looper interface {
	Prepare() error
	Loop(interface{}) (interface{}, error)
	Shutdown()
}

type LocalActor struct {
	LogPrefix    string
	VerboseLevel int
	looper       Looper
	state        int32
	multicker    *Multicker
	controlChan  chan *ControlMsg
	prepareChan  chan error
	stopChan     chan struct{}
}

func NewLocalActor(looper Looper, verboseLevel int) *LocalActor {
	return &LocalActor{
		VerboseLevel: verboseLevel,
		looper:       looper,
		state:        ActorStopped,
		multicker:    NewMulticker(verboseLevel),
		prepareChan:  make(chan error),
		stopChan:     make(chan struct{}),
	}
}

func (m *LocalActor) log(lvl int, ts *time.Time, format string, v ...interface{}) {
	if m.VerboseLevel >= lvl {
		tsStr := ""
		if ts != nil {
			tsStr = ", " + time.Since(*ts).String()
		}
		log.Printf(m.LogPrefix+format+tsStr, v...)
	}
}

func (m *LocalActor) Start() error {
	if !atomic.CompareAndSwapInt32(&m.state, ActorStopped, ActorStarting) {
		return fmt.Errorf("manager is not stopped")
	}
	m.log(1, nil, "starting manager")
	go m.loop()
	return <-m.prepareChan
}

func (m *LocalActor) Stop() error {
	if !atomic.CompareAndSwapInt32(&m.state, ActorStarted, ActorStopping) {
		return fmt.Errorf("manager is not started")
	}
	m.log(1, nil, "stopping manager")
	m.stopChan <- struct{}{}
	return nil
}

func (m *LocalActor) StopAsync() error {
	if !atomic.CompareAndSwapInt32(&m.state, ActorStarted, ActorStopping) {
		return fmt.Errorf("manager is not started")
	}
	m.log(1, nil, "stopping manager")
	go func() {
		m.stopChan <- struct{}{}
	}()
	return nil
}

func (m *LocalActor) AddTicker(id string, d time.Duration, msg interface{}) error {
	return m.multicker.Add(id, d, msg)
}

func (m *LocalActor) DeleteTicker(id string) error {
	return m.multicker.Delete(id)
}

func (m *LocalActor) Send(msg interface{}, waitResult bool) (resp interface{}, err error) {
	defer func() {
		if r := recover(); r != nil {
			err = fmt.Errorf("manager is stopped")
		}
	}()
	if atomic.LoadInt32(&m.state) != ActorStarted {
		err = fmt.Errorf("manager is not started")
		return
	}
	var resultChan chan *ControlMsg
	reqTime := time.Now()

	if waitResult {
		resultChan = make(chan *ControlMsg, 1)
	}
	cmOut := &ControlMsg{
		Msg:        msg,
		ResultChan: resultChan,
	}
	m.log(2, nil, "sending to loop msg=%v", msg)
	m.controlChan <- cmOut
	if waitResult {
		cmIn := <-resultChan
		resp = cmIn.Msg
		err = cmIn.Error
		m.log(2, &reqTime, "recieved from loop msg=%v, err=%v", resp, err)
	} else {
		m.log(2, &reqTime, "sent to loop msg=%v", msg)
	}
	return
}

func (m *LocalActor) loop() {
	defer atomic.StoreInt32(&m.state, ActorStopped)

	err := m.looper.Prepare()
	m.prepareChan <- err
	if err != nil {
		return
	}
	_ = m.multicker.Start()
	m.controlChan = make(chan *ControlMsg)
	atomic.StoreInt32(&m.state, ActorStarted)

x:
	for {
		select {
		case cmIn := <-m.controlChan:
			var msgOut interface{}
			var err error
			reqTime := time.Now()

			m.log(2, nil, "got in loop msg=%v", cmIn.Msg)
			if atomic.LoadInt32(&m.state) == ActorStarted {
				msgOut, err = m.looper.Loop(cmIn.Msg)
			} else {
				m.log(2, &reqTime, "skipping in loop msg=%v", cmIn.Msg)
				msgOut, err = nil, fmt.Errorf("message skipped in manager: not started")
			}
			if cmIn.ResultChan != nil {
				cmOut := &ControlMsg{
					Msg:   msgOut,
					Error: err,
				}
				m.log(2, &reqTime, "sending from loop msg=%v, err=%v", msgOut, err)
				cmIn.ResultChan <- cmOut
			} else {
				m.log(2, &reqTime, "processed in loop msg=%v", cmIn.Msg)
			}
		case msgIn := <-m.multicker.C:
			if atomic.LoadInt32(&m.state) != ActorStarted {
				m.log(2, nil, "skipping tick in loop msg=%v", msgIn)
				break
			}
			tickTime := time.Now()
			m.log(2, nil, "got tick in loop msg=%v", msgIn)
			msgOut, err := m.looper.Loop(msgIn)
			m.log(2, &tickTime, "processed tick in loop msgIn=%v, msgOut=%v, err=%v", msgIn, msgOut, err)
		case <-m.stopChan:
			break x
		}
	}
	_ = m.multicker.Stop()
	m.looper.Shutdown()

	m.log(1, nil, "closing control channel in loop")
	close(m.controlChan)
}
