package localactor

import (
	"fmt"
	"log"
	"sync"
	"sync/atomic"
	"time"
)

// ==========================================================================================

// Send several ticks to one channel with different messages.
//
// type Multicker interface {
//     C chan interface{}
//     Start() error
//     Stop() error
//     Add(string, time.Duration, interface{}) error
//     Delete(string) error
// }
//
const (
	MultickerStopped int32 = iota
	MultickerStarted
)

type SingleTiker struct {
	d    time.Duration
	msg  interface{}
	stop chan struct{}
}

type Multicker struct {
	LogPrefix    string
	C            chan interface{}
	VerboseLevel int
	data         map[string]*SingleTiker
	mutex        sync.RWMutex
	state        int32
}

func NewMulticker(verboseLevel int) *Multicker {
	return &Multicker{
		LogPrefix:    "[ticker] ",
		C:            make(chan interface{}, 1),
		VerboseLevel: verboseLevel,
		data:         make(map[string]*SingleTiker),
		state:        MultickerStopped,
	}
}

func (m *Multicker) log(lvl int, ts *time.Time, format string, v ...interface{}) {
	if m.VerboseLevel >= lvl {
		tsStr := ""
		if ts != nil {
			tsStr = ", " + time.Since(*ts).String()
		}
		log.Printf(m.LogPrefix+format+tsStr, v...)
	}
}

func (m *Multicker) Start() error {
	m.mutex.Lock()
	defer m.mutex.Unlock()

	if atomic.CompareAndSwapInt32(&m.state, MultickerStopped, MultickerStarted) {
		m.log(1, nil, "starting")
		for id, t := range m.data {
			if t.stop == nil {
				m.log(1, nil, "starting id='%s'", id)
				t.stop = make(chan struct{}, 1)
				go m.ticker(id, t.d, t.msg, t.stop)
			}
		}
		return nil
	}
	return fmt.Errorf("already started")
}

func (m *Multicker) Stop() error {
	m.mutex.Lock()
	defer m.mutex.Unlock()

	if atomic.CompareAndSwapInt32(&m.state, MultickerStarted, MultickerStopped) {
		m.log(1, nil, "stopping")
		for id, t := range m.data {
			if t.stop != nil {
				m.log(1, nil, "stopping id='%s'", id)
				t.stop <- struct{}{}
				t.stop = nil
			}
		}
	}
	return fmt.Errorf("already stopped")
}

func (m *Multicker) Add(id string, d time.Duration, msg interface{}) error {
	m.mutex.Lock()
	defer m.mutex.Unlock()

	if _, ok := m.data[id]; ok {
		return fmt.Errorf("id='%s' already registered", id)
	}
	m.log(1, nil, "adding ticker %s", id)
	t := &SingleTiker{d: d, msg: msg}
	m.data[id] = t

	if atomic.LoadInt32(&m.state) == MultickerStarted {
		m.log(1, nil, "starting ticker %s", id)
		t.stop = make(chan struct{}, 1)
		go m.ticker(id, d, msg, t.stop)
	}
	return nil
}

func (m *Multicker) Delete(id string) error {
	m.mutex.Lock()
	defer m.mutex.Unlock()

	if v, ok := m.data[id]; ok {
		if v.stop != nil {
			m.log(1, nil, "stopping id='%s'", id)
			v.stop <- struct{}{}
			v.stop = nil
		}
		m.log(1, nil, "deleting id='%s'", id)
		delete(m.data, id)
		return nil
	}
	return fmt.Errorf("no id='%s' registered", id)
}

func (m *Multicker) ticker(id string, d time.Duration, msg interface{}, stop chan struct{}) {
	m.log(1, nil, "entering id='%s'", id)
	ticker := time.NewTicker(d)
	defer ticker.Stop()

	for {
		select {
		case <-ticker.C:
			m.C <- msg
		case <-stop:
			m.log(1, nil, "exiting id='%s'", id)
			return
		}
	}
}
