package selector

import (
	"a.yandex-team.ru/solomon/libs/go/color"
	"fmt"
	"regexp"
	"strings"
)

type Selectors struct {
	selectors []Selector
}

type Selector interface {
	Match(labels map[string]string) bool
}

type All struct {
}

func (a All) Match(labels map[string]string) bool {
	return true
}

type Any struct {
	key string
}

func (a Any) Match(labels map[string]string) bool {
	_, prs := labels[a.key]
	return prs
}

type Absent struct {
	key string
}

func (a Absent) Match(labels map[string]string) bool {
	_, prs := labels[a.key]
	return !prs
}

type Exact struct {
	key   string
	value string
}

func (e Exact) Match(labels map[string]string) bool {
	val, prs := labels[e.key]
	if !prs {
		return false
	}

	return val == e.value
}

type NotExact struct {
	key   string
	value string
}

func (e NotExact) Match(labels map[string]string) bool {
	val, prs := labels[e.key]
	if !prs {
		return false
	}

	return val != e.value
}

type Glob struct {
	key    string
	values []string
}

func (g Glob) Match(labels map[string]string) bool {
	val, prs := labels[g.key]
	if !prs {
		return false
	}
	for i := 0; i < len(g.values); i++ {
		match := MatchGlob(g.values[i], val)
		if match {
			return true
		}
	}
	return false
}

type NotGlob struct {
	key    string
	values []string
}

func (g NotGlob) Match(labels map[string]string) bool {
	val, prs := labels[g.key]
	if !prs {
		return false
	}
	for i := 0; i < len(g.values); i++ {
		match := MatchGlob(g.values[i], val)
		if !match {
			return true
		}
	}
	return false
}

type Regexp struct {
	key    string
	regexp *regexp.Regexp
}

func (r Regexp) Match(labels map[string]string) bool {
	for k, v := range labels {
		if k == r.key {
			if r.regexp.MatchString(v) {
				return true
			}
		}
	}
	return false
}

type NotRegexp struct {
	key    string
	regexp *regexp.Regexp
}

func (r NotRegexp) Match(labels map[string]string) bool {
	for k, v := range labels {
		if k == r.key {
			if !r.regexp.MatchString(v) {
				return true
			}
		}
	}
	return false
}

func (s Selectors) Match(labels map[string]string) bool {
	for i := 0; i < len(s.selectors); i++ {
		match := s.selectors[i].Match(labels)
		if !match {
			return false
		}
	}
	return true
}

func ParseSelectors(value string) (*Selectors, error) {
	if value == "" {
		return &Selectors{
			selectors: []Selector{&All{}},
		}, nil
	}
	vals := strings.Split(value, ", ")
	selectors := make([]Selector, len(vals))
	for i := 0; i < len(vals); i++ {
		if strings.HasSuffix(vals[i], "=\"*\"") {
			parts := strings.SplitN(vals[i], "=", 2)
			selectors[i] = &Any{key: parts[0]}
			continue
		}
		if strings.HasSuffix(vals[i], "=\"-\"") {
			parts := strings.SplitN(vals[i], "=", 2)
			selectors[i] = &Absent{key: parts[0]}
			continue
		}
		parts := strings.SplitN(vals[i], "!==", 2)
		if len(parts) == 2 {
			selectors[i] = &NotExact{key: parts[0], value: prepareValue(parts[1])}
			continue
		}
		parts = strings.SplitN(vals[i], "==", 2)
		if len(parts) == 2 {
			selectors[i] = &Exact{key: parts[0], value: prepareValue(parts[1])}
			continue
		}
		parts = strings.SplitN(vals[i], "=~", 2)
		if len(parts) == 2 {
			exp, err := regexp.Compile(prepareValue(parts[1]))
			if err != nil {
				return nil, err
			}
			selectors[i] = &Regexp{key: parts[0], regexp: exp}
			continue
		}
		parts = strings.SplitN(vals[i], "!~", 2)
		if len(parts) == 2 {
			exp, err := regexp.Compile(prepareValue(parts[1]))
			if err != nil {
				return nil, err
			}
			selectors[i] = &NotRegexp{key: parts[0], regexp: exp}
			continue
		}
		parts = strings.SplitN(vals[i], "!=", 2)
		if len(parts) == 2 {
			selectors[i] = &NotGlob{key: parts[0], values: strings.Split(prepareValue(parts[1]), "|")}
			continue
		}
		parts = strings.SplitN(vals[i], "=", 2)
		if len(parts) == 2 {
			selectors[i] = &Glob{key: parts[0], values: strings.Split(prepareValue(parts[1]), "|")}
			continue
		}
		return nil, fmt.Errorf("unsupported selector %s", color.Red(vals[i]))
	}
	return &Selectors{selectors: selectors}, nil
}

func prepareValue(value string) string {
	return strings.Replace(value, "\"", "", -1)
}
