package tracker

import (
	"a.yandex-team.ru/solomon/libs/go/uhttp"
	"context"
	"fmt"
	"time"
)

const (
	baseURL     = "https://st-api.yandex-team.ru/v2"
	httpTimeout = 5 * time.Second
)

type Attachment struct {
	ID        string   `json:"id"`
	Name      string   `json:"name"`
	CreatedAt *ISOTime `json:"createdAt"`
}

type Client struct {
	httpClient *uhttp.Client
}

func NewClient(token string) *Client {
	return &Client{httpClient: uhttp.NewClient(baseURL, "", uhttp.OAuthToken(token), httpTimeout, true)}
}

func (c *Client) Myself(ctx context.Context) (*User, error) {
	req, err := c.httpClient.NewGetRequest(ctx, "/myself")
	if err != nil {
		return nil, err
	}

	var user User
	err = c.httpClient.SendJSONRequest(req, &user)
	if err != nil {
		return nil, err
	}
	return &user, nil
}

func (c *Client) GetIssue(ctx context.Context, key string) (*Issue, error) {
	req, err := c.httpClient.NewGetRequest(ctx, "/issues/"+key)
	if err != nil {
		return nil, err
	}

	var issue Issue
	err = c.httpClient.SendJSONRequest(req, &issue)
	if err != nil {
		return nil, err
	}
	return &issue, nil
}

func (c *Client) CreateIssue(ctx context.Context, issue *Issue) error {
	req, err := c.httpClient.NewPostJSONRequest(ctx, "/issues", issue)
	if err != nil {
		return err
	}

	var resp struct {
		Key string `json:"key"`
	}

	if err = c.httpClient.SendJSONRequest(req, &resp); err != nil {
		return err
	}
	key := &resp.Key
	issue.Key = key
	return nil
}

func (c *Client) UpdateIssue(ctx context.Context, issue *Issue) error {
	version := *issue.Version
	issue.Version = nil
	key := *issue.Key
	issue.Key = nil
	req, err := c.httpClient.NewPatchJSONRequest(ctx, fmt.Sprintf("/issues/%s?version=%d", key, version), issue)
	if err != nil {
		return err
	}

	var resp struct {
		Key string `json:"key"`
	}

	if err = c.httpClient.SendJSONRequest(req, &resp); err != nil {
		return err
	}
	return nil
}

func (c *Client) GetCheckListItems(ctx context.Context, issueKey string) ([]CheckListItem, error) {
	req, err := c.httpClient.NewGetRequest(ctx, fmt.Sprintf("/issues/%s/checklistItems", issueKey))
	if err != nil {
		return nil, err
	}

	var checkList []CheckListItem
	if err = c.httpClient.SendJSONRequest(req, &checkList); err != nil {
		return nil, err
	}

	return checkList, nil
}

func (c *Client) SetCheckListItems(ctx context.Context, issueKey string, checkList []CheckListItem) error {
	req, err := c.httpClient.NewPostJSONRequest(ctx, fmt.Sprintf("/issues/%s/checklistItems", issueKey), checkList)
	if err != nil {
		return err
	}

	return c.httpClient.SendJSONRequest(req, nil)
}

func (c *Client) FindIssues(ctx context.Context, filter Filter) ([]Issue, error) {
	req, err := c.httpClient.NewGetRequest(ctx, "/issues?"+filter.String())
	if err != nil {
		return nil, err
	}

	var issues []Issue
	err = c.httpClient.SendJSONRequest(req, &issues)
	if err != nil {
		return nil, err
	}
	return issues, nil
}

func (c *Client) GetLinks(ctx context.Context, issueKey string) ([]Link, error) {
	url := fmt.Sprintf("/issues/%s/links", issueKey)
	req, err := c.httpClient.NewGetRequest(ctx, url)
	if err != nil {
		return nil, err
	}

	var links []Link
	if err = c.httpClient.SendJSONRequest(req, &links); err != nil {
		return nil, err
	}

	return links, nil
}

func (c *Client) CreateLink(ctx context.Context, issueFrom, issueTo string) error {
	url := fmt.Sprintf("/issues/%s/links", issueFrom)
	var reqValue = struct {
		Relationship string `json:"relationship"`
		Issue        string `json:"issue"`
	}{
		Relationship: "relates",
		Issue:        issueTo,
	}

	req, err := c.httpClient.NewPostJSONRequest(ctx, url, &reqValue)
	if err != nil {
		return err
	}

	return c.httpClient.SendJSONRequest(req, nil)
}

func (c *Client) ResolveIssue(ctx context.Context, issueKey string) error {
	url := fmt.Sprintf("/issues/%s/transitions/closed/_execute", issueKey)
	var reqValue = struct {
		Resolution string `json:"resolution"`
	}{
		Resolution: "fixed",
	}

	req, err := c.httpClient.NewPostJSONRequest(ctx, url, &reqValue)
	if err != nil {
		return err
	}

	return c.httpClient.SendJSONRequest(req, nil)
}

func (c *Client) OpenIssue(ctx context.Context, issueKey string) error {
	url := fmt.Sprintf("/issues/%s/transitions/open/_execute", issueKey)
	var reqValue = struct {
	}{}
	req, err := c.httpClient.NewPostJSONRequest(ctx, url, reqValue)
	if err != nil {
		return err
	}

	return c.httpClient.SendJSONRequest(req, nil)
}

func (c *Client) AddAttachment(ctx context.Context, issueKey, name string, body []byte) (string, error) {
	url := fmt.Sprintf("/issues/%s/attachments", issueKey)

	req, err := c.httpClient.NewUploadRequest(ctx, url, name, body, nil)
	if err != nil {
		return "", err
	}
	var respValue = struct {
		ID string `json:"id"`
	}{}
	if err := c.httpClient.SendJSONRequest(req, &respValue); err != nil {
		return "", err
	}
	return respValue.ID, nil
}

func (c *Client) ListAttachments(ctx context.Context, issueKey string) ([]*Attachment, error) {
	req, err := c.httpClient.NewGetRequest(ctx, fmt.Sprintf("/issues/%s/attachments", issueKey))
	if err != nil {
		return nil, err
	}

	var attachmentList []*Attachment
	if err = c.httpClient.SendJSONRequest(req, &attachmentList); err != nil {
		return nil, err
	}
	return attachmentList, nil
}

func (c *Client) GetAttachment(ctx context.Context, issueKey, id string) ([]byte, error) {
	req, err := c.httpClient.NewGetRequest(ctx, fmt.Sprintf("/issues/%s/attachments/%s/", issueKey, id))
	if err != nil {
		return nil, err
	}

	var body []byte
	if body, err = c.httpClient.SendRequest(req); err != nil {
		return nil, err
	}
	return body, nil
}

func (c *Client) RemoveAttachment(ctx context.Context, issueKey, id string) error {
	req, err := c.httpClient.NewDeleteRequest(ctx, fmt.Sprintf("/issues/%s/attachments/%s", issueKey, id))
	if err != nil {
		return err
	}
	if _, err = c.httpClient.SendRequest(req); err != nil {
		return err
	}
	return nil
}
