package tracker

import (
	"encoding/json"
	"time"
)

type Assignee struct {
	ID string `json:"id"`
}

type Component struct {
	ID string `json:"id"`
}

type Queue struct {
	ID  string `json:"id"`
	Key string `json:"key"`
}

type ISOTime time.Time

const (
	isoTimeFormat string = "2006-01-02T15:04:05.000-0700"
)

type Issue struct {
	Key         *string     `json:"key,omitempty"`
	Version     *int64      `json:"version,omitempty"`
	Queue       Queue       `json:"queue"`
	Summary     string      `json:"summary"`
	Description string      `json:"description"`
	Assignee    Assignee    `json:"assignee"`
	Components  []Component `json:"components"`
	CreatedAt   *ISOTime    `json:"createdAt,omitempty"`
}

func (isoTime *ISOTime) UnmarshalJSON(b []byte) error {
	var timeString string
	err := json.Unmarshal(b, &timeString)
	if err != nil {
		return err
	}
	t, err := time.Parse(isoTimeFormat, timeString)
	if err != nil {
		return err
	}
	*isoTime = ISOTime(t)
	return nil
}

func (isoTime ISOTime) MarshalJSON() ([]byte, error) {
	timeString := time.Time(isoTime).Format(isoTimeFormat)
	return json.Marshal(&timeString)
}

func (isoTime ISOTime) String() string {
	return time.Time(isoTime).String()
}

func (isoTime *ISOTime) Format(s string) string {
	return time.Time(*isoTime).Format(s)
}

func (i *Issue) URL() string {
	if i.Key != nil {
		return "https://st.yandex-team.ru/" + *i.Key
	}
	return "unknown URL"
}
