package uhttp

import (
	"bytes"
	"context"
	"encoding/json"
	"fmt"
	"io"
	"io/ioutil"
	"mime/multipart"
	"net/http"
	"net/textproto"
	"net/url"
	"time"
)

const (
	maxRetries = 5
)

type MultiPart struct {
	Headers map[string][]string
	Body    []byte
}

func Query(items ...interface{}) *bytes.Buffer {
	isKey := true
	b := bytes.Buffer{}
	for _, item := range items {
		switch item := item.(type) {
		case string:
			_, _ = b.WriteString(item)
		case []byte:
			_, _ = b.Write(item)
		case byte:
			_ = b.WriteByte(item)
		case rune:
			_, _ = b.WriteRune(item)
		}
		if isKey {
			_ = b.WriteByte('=')
			isKey = false
		} else {
			_ = b.WriteByte('&')
			isKey = true
		}
	}
	if b.Len() > 0 {
		b.Truncate(b.Len() - 1)
	}
	return &b
}

// micro http client
type Client struct {
	baseURL    string
	userAgent  string
	token      Token
	httpClient *http.Client
}

func NewClient(baseURL string, userAgent string, token Token, timeout time.Duration, followRedirects bool) *Client {
	httpClient := &http.Client{Timeout: timeout}
	if !followRedirects {
		httpClient.CheckRedirect =
			func(req *http.Request, via []*http.Request) error {
				return http.ErrUseLastResponse
			}
	}
	return &Client{
		baseURL:    baseURL,
		userAgent:  userAgent,
		token:      token,
		httpClient: httpClient}
}

func (c *Client) fillHeaders(header http.Header, acceptJSON bool) {
	if c.userAgent != "" {
		header.Add("User-Agent", c.userAgent)
	}
	if c.token != nil {
		header.Add("Authorization", c.token.HeaderValue())
	}
	if acceptJSON {
		header.Add("Accept", "application/json")
	}
}

func (c *Client) NewGetRequest(ctx context.Context, url string) (*http.Request, error) {
	req, err := http.NewRequestWithContext(ctx, http.MethodGet, c.baseURL+url, nil)
	if err != nil {
		return nil, fmt.Errorf("cannot create %s request: %v", url, err)
	}
	return req, nil
}

func (c *Client) NewPostRequest(ctx context.Context, url string, reqBody []byte) (*http.Request, error) {
	req, err := http.NewRequestWithContext(ctx, http.MethodPost, c.baseURL+url, bytes.NewBuffer(reqBody))
	if err != nil {
		return nil, fmt.Errorf("cannot create %s request: %v", url, err)
	}
	return req, nil
}

func (c *Client) NewMultipartRequest(ctx context.Context, url string, parts []*MultiPart) (*http.Request, error) {
	body := &bytes.Buffer{}
	mpWriter := multipart.NewWriter(body)

	for _, p := range parts {
		part, err := mpWriter.CreatePart(textproto.MIMEHeader(p.Headers))
		if err != nil {
			return nil, fmt.Errorf("cannot create part of multipart request for %s: %v", url, err)
		}
		if _, err = part.Write(p.Body); err != nil {
			return nil, fmt.Errorf("cannot write part of multipart request for %s: %v", url, err)
		}
	}
	if err := mpWriter.Close(); err != nil {
		return nil, fmt.Errorf("cannot create multipart request for %s: %v", url, err)
	}

	req, err := http.NewRequestWithContext(ctx, http.MethodPost, c.baseURL+url, body)
	if err != nil {
		return nil, fmt.Errorf("cannot create multipart request for %s: %v", url, err)
	}
	req.Header.Set("Content-Type", mpWriter.FormDataContentType())

	return req, nil
}

func (c *Client) NewUploadRequest(ctx context.Context, url, filename string, fileBody []byte, params map[string]string) (*http.Request, error) {
	body := &bytes.Buffer{}
	mpWriter := multipart.NewWriter(body)

	part, err := mpWriter.CreateFormFile("file", filename)
	if err != nil {
		return nil, err
	}
	if _, err = part.Write(fileBody); err != nil {
		return nil, fmt.Errorf("cannot write file body in upload for %s: %v", url, err)
	}
	for key, val := range params {
		if err = mpWriter.WriteField(key, val); err != nil {
			return nil, fmt.Errorf("cannot write parameters in upload for %s: %v", url, err)
		}
	}
	if err := mpWriter.Close(); err != nil {
		return nil, fmt.Errorf("cannot create upload for %s: %v", url, err)
	}

	req, err := http.NewRequestWithContext(ctx, http.MethodPost, c.baseURL+url, body)
	if err != nil {
		return nil, fmt.Errorf("cannot create upload request for %s: %v", url, err)
	}
	req.Header.Set("Content-Type", mpWriter.FormDataContentType())

	return req, nil
}

func (c *Client) NewPutRequest(ctx context.Context, url string, reqBody []byte) (*http.Request, error) {
	req, err := http.NewRequestWithContext(ctx, http.MethodPut, c.baseURL+url, bytes.NewReader(reqBody))
	if err != nil {
		return nil, fmt.Errorf("cannot create %s request: %v", url, err)
	}
	return req, nil
}

func (c *Client) NewDeleteRequest(ctx context.Context, url string) (*http.Request, error) {
	req, err := http.NewRequestWithContext(ctx, http.MethodDelete, c.baseURL+url, nil)
	if err != nil {
		return nil, fmt.Errorf("cannot create %s request: %v", url, err)
	}
	return req, nil
}

func (c *Client) NewPatchRequest(ctx context.Context, url string, reqBody []byte) (*http.Request, error) {
	req, err := http.NewRequestWithContext(ctx, http.MethodPatch, c.baseURL+url, bytes.NewReader(reqBody))
	if err != nil {
		return nil, fmt.Errorf("cannot create %s request: %v", url, err)
	}
	return req, nil
}

func (c *Client) NewPostJSONRequest(ctx context.Context, url string, reqValue interface{}) (*http.Request, error) {
	reqBody, err := json.Marshal(reqValue)
	if err != nil {
		return nil, fmt.Errorf("cannot marshal request body to a JOSN")
	}
	req, err := c.NewPostRequest(ctx, url, reqBody)
	if err != nil {
		return nil, err
	}
	req.Header.Add("Content-Type", "application/json")
	return req, nil
}

func (c *Client) NewPutJSONRequest(ctx context.Context, url string, reqValue interface{}) (*http.Request, error) {
	reqBody, err := json.Marshal(reqValue)
	if err != nil {
		return nil, fmt.Errorf("cannot marshal request body to a JOSN")
	}
	req, err := c.NewPutRequest(ctx, url, reqBody)
	if err != nil {
		return nil, err
	}
	req.Header.Add("Content-Type", "application/json")
	return req, nil
}

func (c *Client) NewPatchJSONRequest(ctx context.Context, url string, reqValue interface{}) (*http.Request, error) {
	reqBody, err := json.Marshal(reqValue)
	if err != nil {
		return nil, fmt.Errorf("cannot marshal request body to a JOSN")
	}
	req, err := c.NewPatchRequest(ctx, url, reqBody)
	if err != nil {
		return nil, err
	}
	req.Header.Add("Content-Type", "application/json")
	return req, nil
}

func (c *Client) sendRequestWithRetries(req *http.Request) (*http.Response, error) {
	// TODO: body is null
	var body []byte
	if req.Body != nil {
		var err error
		body, err = ioutil.ReadAll(req.Body)
		if err != nil {
			return nil, err
		}
		err = req.Body.Close()
		if err != nil {
			return nil, err
		}
	}

	for i := 0; ; i++ {
		reqCopy := req.Clone(req.Context())
		if body != nil {
			reqCopy.Body = ioutil.NopCloser(bytes.NewReader(body))
		}

		resp, err := c.httpClient.Do(reqCopy)
		if err != nil {
			// retry timeout
			if !err.(*url.Error).Timeout() {
				// redirect can produce temporary error
				if resp != nil && resp.StatusCode >= http.StatusMultipleChoices && resp.StatusCode <= http.StatusPermanentRedirect {
					return resp, nil
				}
				// this error already contains all the information about the URL
				return nil, err
			}
		}

		if i > maxRetries {
			return resp, err
		}
		if resp != nil {
			// retry only 429 and 5xx status codes and only if we have more attempts
			if resp.StatusCode != http.StatusTooManyRequests && resp.StatusCode < http.StatusInternalServerError {
				return resp, nil
			}
			// drain unwanted body
			_, _ = io.Copy(io.Discard, resp.Body)
			_ = resp.Body.Close()
		}

		// backoff timeout
		select {
		case <-req.Context().Done():
			return nil, req.Context().Err()
		case <-time.After(nextDelay(i, maxRetries)):
			continue
		}
	}
}

func (c *Client) sendRequest(req *http.Request) ([]byte, error) {
	resp, err := c.sendRequestWithRetries(req)
	if err != nil {
		return nil, fmt.Errorf("cannot send http request: %v", err)
	}
	defer resp.Body.Close()

	respBody, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return nil, fmt.Errorf("cannot read response body: %v", err)
	}

	if resp.StatusCode < http.StatusOK || resp.StatusCode >= http.StatusMultipleChoices {
		return nil, &Error{StatusCode: resp.StatusCode, Message: string(respBody)}
	}

	return respBody, nil
}

func (c *Client) SendRequest(req *http.Request) ([]byte, error) {
	c.fillHeaders(req.Header, false)
	return c.sendRequest(req)
}

func (c *Client) SendRawRequest(req *http.Request) (*http.Response, error) {
	c.fillHeaders(req.Header, false)
	return c.sendRequestWithRetries(req)
}

func (c *Client) SendJSONRequest(req *http.Request, respValue interface{}) error {
	c.fillHeaders(req.Header, true)

	respBody, err := c.sendRequest(req)
	if err != nil {
		return err
	}

	if respValue != nil {
		err = json.Unmarshal(respBody, respValue)
		if err != nil {
			return fmt.Errorf("cannot parse response from a JSON '%s': %v", string(respBody), err)
		}
	}

	return nil
}
