package ussh

import (
	"fmt"
	"golang.org/x/crypto/ssh"
	"io/ioutil"
	"os"
	"path"
)

func newPrivateKeySigner(keyFilename string) (ssh.Signer, error) {
	fd, err := os.Open(keyFilename)
	if err != nil {
		return nil, err
	}
	defer func() { _ = fd.Close() }()

	sshValue, err := ioutil.ReadAll(fd)
	if err != nil {
		return nil, err
	}

	return ssh.ParsePrivateKey(sshValue)
}

func newPublicKeysAuth() ssh.AuthMethod {
	homeDir := os.Getenv("HOME")
	keyFilenames := []string{
		path.Join(homeDir, ".ssh", "id_rsa"),
		path.Join(homeDir, ".ssh", "id_dsa"),
	}

	var signers []ssh.Signer
	for _, keyFilename := range keyFilenames {
		signer, err := newPrivateKeySigner(keyFilename)
		if err != nil {
			if !os.IsNotExist(err) {
				fmt.Fprintf(os.Stderr, "cannot parse private key %s: %v\n", keyFilename, err)
			}
		} else {
			signers = append(signers, signer)
		}
	}

	if len(signers) == 0 {
		return nil
	}

	return ssh.PublicKeys(signers...)
}
