package workerpool

import (
	"fmt"
	"log"
	"sync"
	"time"
)

// ==========================================================================================

// WorkerPool of 'size' Workers doing their job concurrently.
//
// type WorkerPool interface {
//     Do(input []interface{})
//     Stop(sync bool)
// }
//

type workerJob struct {
	input interface{}
	wg    *sync.WaitGroup
}

type WorkerPool struct {
	LogPrefix string
	Size      int
	Debug     bool
	wFunc     Worker
	wg        sync.WaitGroup
	jobChan   chan *workerJob
}

type Worker func(interface{})

func NewWorkerPool(name string, size int, w Worker, debug bool) *WorkerPool {
	f := &WorkerPool{
		LogPrefix: fmt.Sprintf("[%s factory] ", name),
		Size:      size,
		Debug:     debug,
		wFunc:     w,
		jobChan:   make(chan *workerJob, size),
	}
	for idx := 0; idx < size; idx++ {
		go f.worker(idx)
		f.wg.Add(1)
		f.log(nil, "worker #%d started", idx)
	}
	return f
}

func (f *WorkerPool) log(ts *time.Time, format string, v ...interface{}) {
	if f.Debug {
		tsStr := ""
		if ts != nil {
			tsStr = " " + time.Since(*ts).String()
		}
		log.Printf(f.LogPrefix+format+tsStr, v...)
	}
}

func (f *WorkerPool) worker(idx int) {
	for {
		job := <-f.jobChan
		if job.wg == nil {
			break
		}
		reqTime := time.Now()
		f.log(nil, "worker #%d got the job, %v", idx, job.input)
		f.wFunc(job.input)
		job.wg.Done()
		f.log(&reqTime, "worker #%d done the job, %v", idx, job.input)
	}
	f.wg.Done()
	f.log(nil, "worker #%d stopped", idx)
}

func (f *WorkerPool) Do(input []interface{}) {
	wg := new(sync.WaitGroup)
	for _, job := range input {
		wg.Add(1)
		f.jobChan <- &workerJob{
			input: job,
			wg:    wg,
		}
	}
	wg.Wait()
}

func (f *WorkerPool) Stop(sync bool) {
	for idx := 0; idx < f.Size; idx++ {
		f.jobChan <- &workerJob{}
	}
	if sync {
		f.wg.Wait()
	}
loop:
	for {
		select {
		case job := <-f.jobChan:
			job.wg.Done()
			f.log(nil, "while stopping purged job, %v", job.input)
		default:
			break loop
		}
	}
}
